<?php
namespace TPCore\Widgets;

use Elementor\Widget_Base;
use Elementor\Controls_Manager;
use \Elementor\Utils;
use \Elementor\Control_Media;
use \Elementor\Group_Control_Image_Size;
use \Elementor\Group_Control_Border;
use \Elementor\Group_Control_Box_Shadow;
use \Elementor\Group_Control_Text_Shadow;
use \Elementor\Group_Control_Typography;
use \Elementor\Core\Schemes\Typography;
use \Elementor\Group_Control_Background;
use TPCore\Elementor\Controls\Group_Control_TPGradient;
use \Elementor\Repeater;


if (!defined('ABSPATH'))
  exit; // Exit if accessed directly

/**
 * Tp Core
 *
 * Elementor widget for hello world.
 *
 * @since 1.0.0
 */
class TP_Counter extends Widget_Base
{

  use TP_Style_Trait, TP_Icon_Trait, TP_Animation_Trait, TP_Column_Trait;

  /**
   * Retrieve the widget name.
   *
   * @since 1.0.0
   *
   * @access public
   *
   * @return string Widget name.
   */
  public function get_name()
  {
    return 'tp-counter';
  }

  /**
   * Retrieve the widget title.
   *
   * @since 1.0.0
   *
   * @access public
   *
   * @return string Widget title.
   */
  public function get_title()
  {
    return __(TPCORE_THEME_NAME . ' :: Counter', 'tpcore');
  }

  /**
   * Retrieve the widget icon.
   *
   * @since 1.0.0
   *
   * @access public
   *
   * @return string Widget icon.
   */
  public function get_icon()
  {
    return 'tp-icon';
  }

  /**
   * Retrieve the list of categories the widget belongs to.
   *
   * Used to determine where to display the widget in the editor.
   *
   * Note that currently Elementor supports only one category.
   * When multiple categories passed, Elementor uses the first one.
   *
   * @since 1.0.0
   *
   * @access public
   *
   * @return array Widget categories.
   */
  public function get_categories()
  {
    return ['tpcore'];
  }

  /**
   * Retrieve the list of scripts the widget depended on.
   *
   * Used to set scripts dependencies required to run the widget.
   *
   * @since 1.0.0
   *
   * @access public
   *
   * @return array Widget scripts dependencies.
   */
  public function get_script_depends()
  {
    return ['tpcore'];
  }



  /**
   * Register the widget controls.
   *
   * Adds different input fields to allow the user to change and customize the widget settings.
   *
   * @since 1.0.0
   *
   * @access protected
   */

  protected function register_controls()
  {
    $this->register_controls_section();
    $this->style_tab_content();
  }

  protected function register_controls_section()
  {

    $this->tp_design_layout('Layout', 2);

    $this->start_controls_section(
      'tp_counter_sec',
      [
        'label' => esc_html__('Counter', 'tpcore'),
        'tab' => Controls_Manager::TAB_CONTENT,
        'condition' => [
          'tp_design_style' => ['layout-1', 'layout-2', 'layout-3']
        ]
      ]
    );

    $this->add_control(
      'tp_counter_repeater_border_switch',
      [
        'label' => esc_html__('Show Border?', 'tpcore'),
        'type' => Controls_Manager::SWITCHER,
        'label_on' => esc_html__('Show', 'tpcore'),
        'label_off' => esc_html__('Hide', 'tpcore'),
        'return_value' => 'yes',
        'default' => 'yes',
        'condition' => [
          'tp_design_style' => ['layout-1', 'layout-2',]
        ]
      ]
    );

    $this->add_control(
      'tp_counter_icon_switch',
      [
        'label' => esc_html__('Show Icon?', 'tpcore'),
        'type' => Controls_Manager::SWITCHER,
        'label_on' => esc_html__('Show', 'tpcore'),
        'label_off' => esc_html__('Hide', 'tpcore'),
        'return_value' => 'yes',
        'default' => 'yes',
        'condition' => [
          'tp_design_style' => ['layout-2',]
        ]
      ]
    );

    // icon
    $this->tp_single_icon_control('tp_counter', 'tp_counter_icon_switch', 'yes');

    $this->add_control(
      'tp_counter_number',
      [
        'label' => esc_html__('Number', 'tpcore'),
        'type' => Controls_Manager::NUMBER,
        'min' => 0,
        'max' => 1000000,
        'step' => 5,
        'default' => 0,
      ]
    );

    $this->add_control(
      'tp_counter_suffix',
      [
        'label' => esc_html__('Suffix', 'tpcore'),
        'type' => Controls_Manager::TEXT,
        'default' => '%',
      ]
    );
    $this->add_control(
      'tp_counter_prefix',
      [
        'label' => esc_html__('Prefix', 'tpcore'),
        'type' => Controls_Manager::TEXT,
        'default' => '',
      ]
    );

    $this->add_control(
      'tp_counter_duration',
      [
        'label' => esc_html__('Duration', 'tpcore'),
        'type' => Controls_Manager::NUMBER,
        'min' => 0,
        'max' => 100,
        'step' => 1,
        'default' => 1,
      ]
    );


    $this->add_control(
      'tp_counter_title',
      [
        'label' => esc_html__('Title', 'tpcore'),
        'type' => Controls_Manager::TEXTAREA,
        'default' => 'PROJECTS COMPLETED',
        'label_block' => true,
      ]
    );

    $this->add_responsive_control(
      'tp_counter_title_align',
      [
          'label' => esc_html__('Alignment', 'tpcore'),
          'type' => Controls_Manager::CHOOSE,
          'options' => [
              'left' => [
                  'title' => esc_html__('Left', 'tpcore'),
                  'icon' => 'eicon-text-align-left',
              ],
              'center' => [
                  'title' => esc_html__('Center', 'tpcore'),
                  'icon' => 'eicon-text-align-center',
              ],
              'right' => [
                  'title' => esc_html__('Right', 'tpcore'),
                  'icon' => 'eicon-text-align-right',
              ],
          ],
          'default' => 'center',
          'toggle' => true,
          'selectors' => [
              '{{WRAPPER}} .tp_counter_title_align' => 'text-align: {{VALUE}};',
          ],
      ]
  );


    $this->add_control(
      'tp_counter_title_title_tag',
      [
        'label' => esc_html__('Title HTML Tag', 'tpcore'),
        'type' => Controls_Manager::SELECT,
        'options' => [
          'h1' => 'H1',
          'h2' => 'H2',
          'h3' => 'H3',
          'h4' => 'H4',
          'h5' => 'H5',
          'h6' => 'H6',
          'div' => 'div',
          'span' => 'span',
          'p' => 'p',
        ],
        'default' => 'div',
        'condition' => [
          'tp_counter_title!' => '',
        ],
      ]
    );

    $this->end_controls_section();

    // animation
    $this->tp_creative_animation('layout-2');


    $this->tp_columns('col', ['layout-40']);

  }

  // style_tab_content
  protected function style_tab_content()
  {

    $this->tp_section_style_controls('tp_counter_section', 'Section', '.tp-el-section');
    $this->tp_icon_style('layout-2', 'tp_counter_icon', '.tp-icon-box-icon', 'Icon/Image/SVG', true);
    $this->tp_basic_style_controls('tp_counter_number', 'Number', '.tp-el-counter-number');
    $this->tp_basic_style_controls('tp_counter_suffix', 'Suffix', '.tp-el-counter-suffix');
    $this->tp_basic_style_controls('tp_counter_title', 'Title', '.tp-el-counter-title');

  }

  /**
   * Render the widget output on the frontend.
   *
   * Written in PHP and used to generate the final HTML.
   *
   * @since 1.0.0
   *
   * @access protected
   */
  protected function render()
  {
    $settings = $this->get_settings_for_display();

    $tag = $settings['tp_counter_title_title_tag'];

    $attrs = [
      'data-purecounter-duration' => $settings['tp_counter_duration'],
      'data-purecounter-end' => $settings['tp_counter_number'],
      'class' => 'tp-el-counter-number purecounter',
    ];

    ?>

    <?php if ($settings['tp_design_style'] == 'layout-2'):
      $animation = $this->tp_animation_show($settings);
      ?>

      <div class="tp-counter-4-item text-center <?php echo esc_attr($animation['animation']); ?>" <?php echo $animation['duration'] . ' ' . $animation['delay']; ?>>
        <div class="tp-counter-4-icon tp-icon-box-icon">
          <?php tp_render_signle_icon_html($settings, 'tp_counter'); ?>
        </div>
        <div class="tp-counter-4-content">

          <h3 class="tp-counter-4-count mb-10">
            <?php if (!empty($settings['tp_counter_prefix'])): ?>
              <span>
                <?php echo tp_kses($settings['tp_counter_prefix']); ?>
              </span>
            <?php endif; ?>

            <i <?php echo tp_implode_html_attributes($attrs); ?>>0</i>

            <?php if (!empty($settings['tp_counter_suffix'])): ?>
              <span class="tp-el-counter-suffix">
                <?php echo tp_kses($settings['tp_counter_suffix']); ?>
              </span>
            <?php endif; ?>
          </h3>

          <?php if (!empty($settings['tp_counter_repeater_border_switch'])): ?>
            <div class="tp-counter-4-border"></div>
          <?php endif; ?>

          <?php if (!empty($settings['tp_counter_title'])) {
            printf('<%1$s class="%3$s">%2$s</%1$s>', $tag, $settings['tp_counter_title'], 'tp-el-counter-title');
          }
          ?>
        </div>
      </div>

    <?php elseif ($settings['tp_design_style'] == 'layout-3'):
      $animation = $this->tp_animation_show($settings);
      // shape image
      if (!empty($settings['tp_shape_image_1']['url'])) {
        $tp_shape_image = !empty($settings['tp_shape_image_1']['id']) ? wp_get_attachment_image_url($settings['tp_shape_image_1']['id'], $settings['shape_image_size_size']) : $settings['tp_shape_image_1']['url'];
        $tp_shape_image_alt = get_post_meta($settings["tp_shape_image_1"]["id"], "_wp_attachment_image_alt", true);
      }

      if (!empty($settings['tp_shape_image_2']['url'])) {
        $tp_shape_image2 = !empty($settings['tp_shape_image_2']['id']) ? wp_get_attachment_image_url($settings['tp_shape_image_2']['id'], $settings['shape_image_size_size']) : $settings['tp_shape_image_2']['url'];
        $tp_shape_image_alt2 = get_post_meta($settings["tp_shape_image_2"]["id"], "_wp_attachment_image_alt", true);
      }
      ?>

      <div
        class="tp-hero-4-achievement <?php echo esc_attr($animation['animation']); ?> black-bg-2 position-static tp-el-section p-2">
        <?php if (!empty($tp_shape_image)): ?>
          <div class="tp-hero-4-star-1">
            <img class="tp-zoom-in-out" src="<?php echo esc_url($tp_shape_image); ?>"
              alt="<?php echo esc_attr($tp_shape_image_alt); ?>">
          </div>
        <?php endif; ?>
        <?php if (!empty($tp_shape_image2)): ?>
          <div class="tp-hero-4-star-2">
            <img class="tp-zoom-in-out-2" src="<?php echo esc_url($tp_shape_image2); ?>"
              alt="<?php echo esc_attr($tp_shape_image_alt2); ?>">
          </div>
        <?php endif; ?>
        <span><?php echo tp_kses($settings['tp_counter_prefix']); ?><i <?php echo tp_implode_html_attributes($attrs); ?>>0</i><?php echo tp_kses($settings['tp_counter_suffix']); ?></span>
        <?php
        if (!empty($settings['tp_counter_title'])) {
          printf('<%1$s class="%3$s">%2$s</%1$s>', $tag, $settings['tp_counter_title'], 'tp-el-counter-title');
        }
        ?>
      </div>

    <?php elseif ($settings['tp_design_style'] == 'layout-4'):
      $animation = $this->tp_animation_show($settings);

      $hide_border = $settings['tp_counter_repeater_border_switch'] == 'yes' ? ' ' : ' hide-funfact-border';

      ?>

      <!-- counter area start -->
      <div
        class="slide-funfact-height slide-funfact p-relative d-flex align-items-center justify-content-center <?php echo esc_attr($hide_border); ?>">
        <div class="img-marq slide-funfact-overlay">
          <div class="middle-shadow">
            <span></span>
          </div>

          <div class="slide-img-left">
            <div class="box">

              <?php foreach ($settings['tp_counter_shape_left_rep_slides'] as $key => $item):
                $tp_left_img = tp_get_img($item, 'tp_counter_shape_left_rep_image', 'full', false);
                extract($tp_left_img);

                if (!empty($tp_counter_shape_left_rep_image)):
                  ?>
                  <img src="<?php echo esc_url($tp_counter_shape_left_rep_image); ?>"
                    alt="<?php echo esc_attr($tp_counter_shape_left_rep_image_alt); ?>">
                <?php endif; endforeach; ?>

            </div>
            <div class="box">

              <?php foreach ($settings['tp_counter_shape_left_rep_slides'] as $key => $item):
                $tp_left_img = tp_get_img($item, 'tp_counter_shape_left_rep_image', 'full', false);
                extract($tp_left_img);

                if (!empty($tp_counter_shape_left_rep_image)):
                  ?>
                  <img src="<?php echo esc_url($tp_counter_shape_left_rep_image); ?>"
                    alt="<?php echo esc_attr($tp_counter_shape_left_rep_image_alt); ?>">
                <?php endif; endforeach; ?>

            </div>
          </div>

          <div class="slide-img-right">
            <div class="box">

              <?php foreach ($settings['tp_counter_shape_right_rep_slides'] as $key => $item):
                $tp_right_img = tp_get_img($item, 'tp_counter_shape_right_rep_image', 'full', false);
                extract($tp_right_img);

                if (!empty($tp_counter_shape_right_rep_image)):
                  ?>
                  <img src="<?php echo esc_url($tp_counter_shape_right_rep_image); ?>"
                    alt="<?php echo esc_attr($tp_counter_shape_right_rep_image_alt); ?>">
                <?php endif; endforeach; ?>

            </div>
            <div class="box">

              <?php foreach ($settings['tp_counter_shape_right_rep_slides'] as $key => $item):
                $tp_right_img = tp_get_img($item, 'tp_counter_shape_right_rep_image', 'full', false);
                extract($tp_right_img);

                if (!empty($tp_counter_shape_right_rep_image)):
                  ?>
                  <img src="<?php echo esc_url($tp_counter_shape_right_rep_image); ?>"
                    alt="<?php echo esc_attr($tp_counter_shape_right_rep_image_alt); ?>">
                <?php endif; endforeach; ?>

            </div>
          </div>

        </div>
        <div class="slide-funfact-wrap">
          <div>
            <div class="container">
              <div class="row <?php echo esc_attr($this->col_show($settings)); ?>">
                <?php foreach ($settings['tp_counter_rep_list'] as $item):

                  $tag = $item['tp_counter_rep_title_tag'];
                  $title = $item['tp_counter_rep_title'];

                  $attrs = [
                    'data-purecounter-duration' => $item['tp_counter_rep_duration'],
                    'data-purecounter-end' => $item['tp_counter_rep_number'],
                    'class' => 'tp-el-counter-number purecounter',
                  ];
                  ?>

                  <div class="col mb-30">
                    <div class="slide-funfact-item text-center">
                      <h4>
                        <?php echo tp_kses($item['tp_counter_rep_prefix']); ?>
                        <i <?php echo tp_implode_html_attributes($attrs); ?>>0</i>
                        <?php echo tp_kses($item['tp_counter_rep_suffix']); ?>
                      </h4>

                      <?php
                      if (!empty($title)) {
                        printf('<%1$s class="%3$s">%2$s</%1$s>', $tag, $title, 'tp-el-counter-title');
                      }
                      ?>
                    </div>
                  </div>
                <?php endforeach; ?>

              </div>
            </div>
          </div>
        </div>
      </div>
      <!-- counter area end -->

    <?php else: ?>

      <div class="tp-counter-item text-center tp-el-section">
        <?php if (!empty($settings['tp_counter_repeater_border_switch'])): ?>
          <span class="tp-acadia-counter-border"></span>
        <?php endif; ?>
        <h3 class="tp-counter-count mb-10 tp_counter_title_align">
          <?php if (!empty($settings['tp_counter_prefix'])): ?>
            <span>
              <?php echo tp_kses($settings['tp_counter_prefix']); ?>
            </span>
          <?php endif; ?>

          <i <?php echo tp_implode_html_attributes($attrs); ?>>0</i>

          <?php if (!empty($settings['tp_counter_suffix'])): ?>
            <span class="tp-el-counter-suffix">
              <?php echo tp_kses($settings['tp_counter_suffix']); ?>
            </span>
          <?php endif; ?>
        </h3>

        <?php if (!empty($settings['tp_counter_title'])) {
          printf('<%1$s class="%3$s">%2$s</%1$s>', $tag, $settings['tp_counter_title'], 'tp-el-counter-title tp_counter_title_align');
        }
        ?>
      </div>

    <?php endif; ?>

    <?php

  }
}

$widgets_manager->register(new TP_Counter());