<?php

namespace TPCore\Widgets;

use Elementor\Widget_Base;
use Elementor\Controls_Manager;
use \Elementor\Utils;
use \Elementor\Control_Media;

use \Elementor\Group_Control_Border;
use \Elementor\Group_Control_Box_Shadow;
use \Elementor\Group_Control_Text_Shadow;
use \Elementor\Group_Control_Typography;
use \Elementor\Core\Schemes\Typography;
use \Elementor\Group_Control_Background;
use \Elementor\Group_Control_Image_Size;

if (!defined('ABSPATH')) exit; // Exit if accessed directly

/**
 * Tp Core
 *
 * Elementor widget for hello world.
 *
 * @since 1.0.0
 */
class TP_Image extends Widget_Base
{

	use \TPCore\Widgets\TP_Style_Trait;
	use \TPCore\Widgets\TP_Animation_Trait;

	/**
	 * Retrieve the widget name.
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 *
	 * @return string Widget name.
	 */
	public function get_name()
	{
		return 'tp-image';
	}

	/**
	 * Retrieve the widget title.
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 *
	 * @return string Widget title.
	 */
	public function get_title()
	{
		return __(TPCORE_THEME_NAME . ' :: Image', 'tpcore');
	}

	/**
	 * Retrieve the widget icon.
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 *
	 * @return string Widget icon.
	 */
	public function get_icon()
	{
		return 'tp-icon';
	}

	/**
	 * Retrieve the list of categories the widget belongs to.
	 *
	 * Used to determine where to display the widget in the editor.
	 *
	 * Note that currently Elementor supports only one category.
	 * When multiple categories passed, Elementor uses the first one.
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 *
	 * @return array Widget categories.
	 */
	public function get_categories()
	{
		return ['tpcore'];
	}

	/**
	 * Retrieve the list of scripts the widget depended on.
	 *
	 * Used to set scripts dependencies required to run the widget.
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 *
	 * @return array Widget scripts dependencies.
	 */
	public function get_script_depends()
	{
		return ['tpcore'];
	}

	/**
	 * Register the widget controls.
	 *
	 * Adds different input fields to allow the user to change and customize the widget settings.
	 *
	 * @since 1.0.0
	 *
	 * @access protected
	 */

	protected function register_controls()
	{
		$this->register_controls_section();
		$this->style_tab_content();
	}

	protected function register_controls_section()
	{

		// layout Panel
		$this->tp_design_layout('Layout Style', 1);

		// image
		$this->start_controls_section(
			'thumb_section',
			[
				'label' => esc_html__('Image Section', 'tpcore'),
				'condition' => [
					'tp_design_style' => ['layout-1', 'layout-2', 'layout-3'],
				]
			]
		);

		$this->add_control(
			'tp_image',
			[
				'label' => esc_html__('Choose Image', 'tpcore'),
				'type' => \Elementor\Controls_Manager::MEDIA,
				'default' => [
					'url' => \Elementor\Utils::get_placeholder_image_src(),
				],
			]
		);

		$this->add_group_control(
			Group_Control_Image_Size::get_type(),
			[
				'name' => 'tp_image_size',
				'default' => 'full',
				'exclude' => [
					'custom'
				]
			]
		);

		$this->end_controls_section();


		$this->start_controls_section(
			'tp_img_section',
			[
				'label' => esc_html__('Image Controls', 'tpcore'),
				'tab'   => \Elementor\Controls_Manager::TAB_CONTENT,
				'condition' => [
					'tp_design_style' => 'layout-4',
				]
			]
		);

		$this->add_control(
			'tp_img',
			[
				'label' => esc_html__('Choose Image', 'tpcore'),
				'type' => \Elementor\Controls_Manager::MEDIA,
				'default' => [
					'url' => \Elementor\Utils::get_placeholder_image_src(),
				],
			]
		);

		tp_render_links_controls($this, 'img');

		$this->end_controls_section();

		$this->start_controls_section(
			'tp_video_section',
			[
				'label' => esc_html__('Video Controls', 'tpcore'),
				'tab'   => \Elementor\Controls_Manager::TAB_CONTENT,
				'condition' => [
					'tp_design_style' => 'layout-5',
				]
			]
		);

		$this->add_control(
			'tp_video_url',
			[
				'label'       => esc_html__('URL', 'tpcore'),
				'type'        => \Elementor\Controls_Manager::TEXT,
				'default'     => esc_html__('yoururl.com/video', 'tpcore'),
				'placeholder' => esc_html__('Your URL', 'tpcore'),
				'label_block' => true
			]
		);


		$this->end_controls_section();
	}

	protected function style_tab_content()
	{

		$this->start_controls_section(
			'tp_image_style_sec',
			[
				'label' => esc_html__('Image Styles', 'tpcore'),
				'tab'   => \Elementor\Controls_Manager::TAB_STYLE,

			]
		);

		$this->add_control(
			'tp_image_style_width',
			[
				'label' => esc_html__('Width', 'tpcore'),
				'type' => \Elementor\Controls_Manager::SLIDER,
				'size_units' => ['px', '%', 'em', 'rem', 'custom'],
				'selectors' => [
					'{{WRAPPER}} .tp-el-image img' => 'width: {{SIZE}}{{UNIT}};',
				],
			]
		);
		$this->add_control(
			'tp_image_style_height',
			[
				'label' => esc_html__('Height', 'tpcore'),
				'type' => \Elementor\Controls_Manager::SLIDER,
				'size_units' => ['px', '%', 'em', 'rem', 'custom'],
				'selectors' => [
					'{{WRAPPER}} .tp-el-image img' => 'height: {{SIZE}}{{UNIT}};',
				],
			]
		);

		$this->add_control(
			'tp_image_object_style',
			[
				'label' => esc_html__('Object Fit', 'tpcore'),
				'type' => \Elementor\Controls_Manager::SELECT,
				'default' => 'cover',
				'options' => [
					'none' => esc_html__('None', 'tpcore'),
					'contain'  => esc_html__('Contain', 'tpcore'),
					'cover' => esc_html__('Cover', 'tpcore'),
					'fill' => esc_html__('fill', 'tpcore'),
				],
				'selectors' => [
					'{{WRAPPER}} .tp-el-image img' => 'object-fit: {{VALUE}};',
				],
			]
		);
		$this->end_controls_section();


		$this->start_controls_section(
			'tp_video_style_sec',
			[
				'label' => esc_html__('Video Styles', 'tpcore'),
				'tab'   => \Elementor\Controls_Manager::TAB_STYLE,
				'condition' => [
					'tp_design_style' => 'layout-5',
				]
			]
		);

		$this->add_control(
			'tp_video_style_width',
			[
				'label' => esc_html__('Width', 'tpcore'),
				'type' => \Elementor\Controls_Manager::SLIDER,
				'size_units' => ['px', '%', 'em', 'rem', 'custom'],
				'range' => [
					'px' => [
						'min' => 0,
						'max' => 1000,
						'step' => 5,
					],
					'%' => [
						'min' => 0,
						'max' => 100,
					],
				],
				'default' => [
					'unit' => '%',
					'size' => 100,
				],
				'selectors' => [
					'{{WRAPPER}} .tp-el-video' => 'width: {{SIZE}}{{UNIT}};',
				],
			]
		);
		$this->add_control(
			'tp_video_style_height',
			[
				'label' => esc_html__('Height', 'tpcore'),
				'type' => \Elementor\Controls_Manager::SLIDER,
				'size_units' => ['px', '%', 'em', 'rem', 'custom'],
				'range' => [
					'px' => [
						'min' => 0,
						'max' => 1000,
						'step' => 5,
					],
					'%' => [
						'min' => 0,
						'max' => 100,
					],
				],
				'default' => [
					'unit' => 'px',
					'size' => 230,
				],
				'selectors' => [
					'{{WRAPPER}} .tp-el-video' => 'height: {{SIZE}}{{UNIT}};',
				],
			]
		);
		$this->end_controls_section();
	}

	/**
	 * Render the widget output on the frontend.
	 *
	 * Written in PHP and used to generate the final HTML.
	 *
	 * @since 1.0.0
	 *
	 * @access protected
	 */
	protected function render()
	{
		$settings = $this->get_settings_for_display();
?>

		<?php if ($settings['tp_design_style']  == 'layout-2') :

			$img = tp_get_img($settings, 'tp_image', 'tp_image_size');

		?>

			<div class="tp-shop-banner-anim">
				<img data-speed="<?php echo esc_attr($settings['tp_data_speed']); ?>" class="w-100" src="<?php echo esc_url($img['tp_image']); ?>" alt="<?php echo esc_url($img['tp_image_alt']); ?>">
			</div>


		<?php elseif ($settings['tp_design_style'] == 'layout-3') :
			$img = tp_get_img($settings, 'tp_image', 'tp_image_size');
		?>

			<img data-speed="<?php echo esc_attr($settings['tp_data_speed']); ?>" src="<?php echo esc_url($img['tp_image']); ?>" alt="<?php echo esc_url($img['tp_image_alt']); ?>">

		<?php elseif ($settings['tp_design_style'] == 'layout-4') :
			$img = tp_get_img($settings, 'tp_img', 'full', false);

			$attrs = tp_get_repeater_links_attr($settings, 'img');
			extract($attrs);

			$links_attrs = [
				'href' => $link,
				'target' => $target,
				'rel' => $rel
			];
		?>

			<div class="acadia-thumb-box">
				<?php if (!empty($link)) : ?>
					<a <?php echo tp_implode_html_attributes($links_attrs); ?>>
						<img src="<?php echo esc_url($img['tp_img']); ?>" alt="<?php echo esc_url($img['tp_img_alt']); ?>">
					</a>
				<?php else : ?>
					<img src="<?php echo esc_url($img['tp_img']); ?>" alt="<?php echo esc_url($img['tp_img_alt']); ?>">
				<?php endif; ?>
			</div>

		<?php elseif ($settings['tp_design_style'] == 'layout-5') : ?>

			<div class="acadia-video-box">
				<video class="play-video tp-el-video object-fit-cover" loop="" muted="" autoplay="" playsinline="">
					<source src="<?php echo esc_attr($settings['tp_video_url']); ?>" type="video/mp4">
				</video>
			</div>

		<?php else :
			// thumbnail
			$img = tp_get_img($settings, 'tp_image', 'tp_image_size');
		?>

			<div class="offcanvas-gallery-img fix tp-el-image">
				<img src="<?php echo esc_url($img['tp_image']); ?>" alt="<?php echo esc_url($img['tp_image_alt']); ?>">
			</div>

<?php endif;
	}
}

$widgets_manager->register(new TP_Image());
