<?php
namespace TPCore\Widgets;

use Elementor\Widget_Base;
use Elementor\Controls_Manager;
use \Elementor\Utils;
use \Elementor\Control_Media;
use \Elementor\Group_Control_Image_Size;

use \Elementor\Group_Control_Border;
use \Elementor\Group_Control_Box_Shadow;
use \Elementor\Group_Control_Text_Shadow;
use \Elementor\Group_Control_Typography;
use \Elementor\Core\Schemes\Typography;
use \Elementor\Group_Control_Background;
use \Elementor\Repeater;

if (!defined('ABSPATH'))
    exit; // Exit if accessed directly

/**
 * Tp Core
 *
 * Elementor widget for hello world.
 *
 * @since 1.0.0
 */
class TP_Testimonial_Slider extends Widget_Base
{

    use \TPCore\Widgets\TP_Style_Trait;
    use \TPCore\Widgets\TP_Animation_Trait;
    use \TPCore\Widgets\TP_Icon_Trait;

    /**
     * Retrieve the widget name.
     *
     * @since 1.0.0
     *
     * @access public
     *
     * @return string Widget name.
     */
    public function get_name()
    {
        return 'tp-testimonial-slider';
    }

    /**
     * Retrieve the widget title.
     *
     * @since 1.0.0
     *
     * @access public
     *
     * @return string Widget title.
     */
    public function get_title()
    {
        return __(TPCORE_THEME_NAME . ' :: Testimonial Slider', 'tpcore');
    }

    /**
     * Retrieve the widget icon.
     *
     * @since 1.0.0
     *
     * @access public
     *
     * @return string Widget icon.
     */
    public function get_icon()
    {
        return 'tp-icon';
    }

    /**
     * Retrieve the list of categories the widget belongs to.
     *
     * Used to determine where to display the widget in the editor.
     *
     * Note that currently Elementor supports only one category.
     * When multiple categories passed, Elementor uses the first one.
     *
     * @since 1.0.0
     *
     * @access public
     *
     * @return array Widget categories.
     */
    public function get_categories()
    {
        return ['tpcore'];
    }

    /**
     * Retrieve the list of scripts the widget depended on.
     *
     * Used to set scripts dependencies required to run the widget.
     *
     * @since 1.0.0
     *
     * @access public
     *
     * @return array Widget scripts dependencies.
     */
    public function get_script_depends()
    {
        return ['tpcore'];
    }

    /**
     * Register the widget controls.
     *
     * Adds different input fields to allow the user to change and customize the widget settings.
     *
     * @since 1.0.0
     *
     * @access protected
     */

    protected function register_controls()
    {
        $this->register_controls_section();
        $this->style_tab_content();
    }

    protected function register_controls_section()
    {

        // layout Panel
        $this->tp_design_layout('Layout Style', 2);

        // testimonial content section
        $this->start_controls_section(
            'tp_testimonial_content_section',
            [
                'label' => __('Testimonial Content', 'tpcore'),
                'tab' => Controls_Manager::TAB_CONTENT,
                'condition' => [
                    'tp_design_style' => ['layout-1']
                ]
            ]
        );

        $this->add_control(
            'tp_subtitle',
            [
                'label' => esc_html__('Section Subtitle', 'tpcore'),
                'type' => Controls_Manager::TEXT,
                'default' => esc_html__('Subtitle', 'tpcore'),
                'label_block' => true,
            ]
        );

        $this->add_control(
            'tp_title',
            [
                'label' => esc_html__('Section Title', 'tpcore'),
                'type' => Controls_Manager::TEXT,
                'default' => esc_html__('Section Title', 'tpcore'),
                'label_block' => true,
            ]
        );

        $this->add_control(
            'tp_box_icon_type',
            [
                'label' => esc_html__('Select Quote Icon Type', 'tpcore'),
                'type' => \Elementor\Controls_Manager::SELECT,
                'default' => 'icon',
                'options' => [
                    'image' => esc_html__('Image', 'tpcore'),
                    'icon' => esc_html__('Icon', 'tpcore'),
                    'svg' => esc_html__('SVG', 'tpcore'),
                ],
            ]
        );
        $this->add_control(
            'tp_box_icon_svg',
            [
                'show_label' => false,
                'type' => Controls_Manager::TEXTAREA,
                'label_block' => true,
                'placeholder' => esc_html__('SVG/Text Code Here', 'tpcore'),
                'condition' => [
                    'tp_box_icon_type' => 'svg',
                ]
            ]
        );

        $this->add_control(
            'tp_box_icon_image',
            [
                'label' => esc_html__('Upload Icon Image', 'tpcore'),
                'type' => Controls_Manager::MEDIA,
                'default' => [
                    'url' => Utils::get_placeholder_image_src(),
                ],
                'condition' => [
                    'tp_box_icon_type' => 'image',
                ]
            ]
        );

        if (tp_is_elementor_version('<', '2.6.0')) {
            $this->add_control(
                'tp_box_icon',
                [
                    'show_label' => false,
                    'type' => Controls_Manager::ICON,
                    'label_block' => true,
                    'default' => 'fa fa-star',
                    'condition' => [
                        'tp_box_icon_type' => 'icon',
                    ]
                ]
            );
        } else {
            $this->add_control(
                'tp_box_selected_icon',
                [
                    'show_label' => false,
                    'type' => Controls_Manager::ICONS,
                    'fa4compatibility' => 'icons',
                    'label_block' => true,
                    'default' => [
                        'value' => 'fas fa-star',
                        'library' => 'solid',
                    ],
                    'condition' => [
                        'tp_box_icon_type' => 'icon',
                    ]
                ]
            );
        }

        $this->end_controls_section();

        // testimonial section
        $this->start_controls_section(
            'tp_testimonial_section',
            [
                'label' => __('Testimonial Repeater Item', 'tpcore'),
                'tab' => Controls_Manager::TAB_CONTENT,
            ]
        );

        $repeater = new Repeater();

        $repeater->add_control(
            'repeater_condition',
            [
                'label' => __('Field condition', 'tpcore'),
                'type' => Controls_Manager::SELECT,
                'options' => [
                    'style_1' => __('Style 1', 'tpcore'),
                    'style_2' => __('Style 2', 'tpcore'),
                ],
                'default' => 'style_1',
                'frontend_available' => true,
                'style_transfer' => true,
            ]
        );

        $repeater->add_control(
            'tp_testi_rating',
            [
                'label' => esc_html__('Select Rating Count', 'tpcore'),
                'type' => Controls_Manager::SELECT,
                'options' => [
                    '1' => esc_html__('Single Star', 'tpcore'),
                    '2' => esc_html__('2 Star', 'tpcore'),
                    '3' => esc_html__('3 Star', 'tpcore'),
                    '4' => esc_html__('4 Star', 'tpcore'),
                    '5' => esc_html__('5 Star', 'tpcore'),
                ],
                'default' => '5',
                'condition' => [
                    'repeater_condition' => ['style_2', 'style_40']
                ]
            ]
        );

        $repeater->add_control(
            'testimonial_logo',
            [
                'label' => esc_html__('Testimonial Avatar', 'tpcore'),
                'type' => Controls_Manager::MEDIA,
                'default' => [
                    'url' => Utils::get_placeholder_image_src(),
                ],
            ]
        );

        $repeater->add_control(
            'testimonial_content',
            [
                'label' => esc_html__('Content', 'tpcore'),
                'type' => Controls_Manager::TEXTAREA,
                'default' => esc_html__('Nulla ut enim non magna placerat scelerisque sed eu dolor. Sed eu faucibus turpis. Ut bibendum tempor tempus. Ut scelerisque est posuere ex pretium laoreet.', 'tpcore'),
                'description' => tp_get_allowed_html_desc('intermediate'),
                'label_block' => true,
            ]
        );

        $repeater->add_control(
            'testimonial_name',
            [
                'label' => esc_html__('Name', 'tpcore'),
                'type' => Controls_Manager::TEXT,
                'default' => esc_html__('David Prutra', 'tpcore'),
                'description' => tp_get_allowed_html_desc('intermediate'),
                'label_block' => true,
            ]
        );

        $repeater->add_control(
            'testimonial_position',
            [
                'label' => esc_html__('Position', 'tpcore'),
                'type' => Controls_Manager::TEXT,
                'default' => esc_html__('Designer', 'tpcore'),
                'description' => tp_get_allowed_html_desc('intermediate'),
                'label_block' => true,
            ]
        );

        $repeater->add_control(
            'testimonial_shape_img',
            [
                'label' => esc_html__('Shape Image 1', 'tpcore'),
                'type' => Controls_Manager::MEDIA,
                'default' => [
                    'url' => Utils::get_placeholder_image_src(),
                ],
                'condition' => [
                    'repeater_condition' => ['style_2', 'style_40']
                ]
            ]
        );

        $repeater->add_control(
            'testimonial_shape_img2',
            [
                'label' => esc_html__('Shape Image 2', 'tpcore'),
                'type' => Controls_Manager::MEDIA,
                'default' => [
                    'url' => Utils::get_placeholder_image_src(),
                ],
                'condition' => [
                    'repeater_condition' => ['style_2', 'style_40']
                ]
            ]
        );

        $this->add_control(
            'tp_testimonial_slides',
            [
                'show_label' => false,
                'type' => Controls_Manager::REPEATER,
                'fields' => $repeater->get_controls(),
                'title_field' => esc_html__('Testimonial Item', 'tpcore'),
                'default' => [
                    [
                        'testimonial_name' => __('Stifin Shkaya', 'tpcore')
                    ],
                    [
                        'testimonial_name' => __('Jhone jaouy', 'tpcore')
                    ],
                ]
            ]
        );

        $this->end_controls_section();

        // tp_testimonial_vedio_section
        $this->start_controls_section(
            'tp_testimonial_vedio_section',
            [
                'label' => __('Vedio Content', 'tpcore'),
                'tab' => Controls_Manager::TAB_CONTENT,
                'condition' => [
                    'tp_design_style' => ['layout-1']
                ]
            ]
        );

        $this->add_control(
            'tp_vedio_title',
            [
                'label' => esc_html__('Vedio Title', 'tpcore'),
                'type' => Controls_Manager::TEXT,
                'default' => esc_html__('Vedio Title', 'tpcore'),
                'label_block' => true,
            ]
        );

        $this->add_control(
            'tp_tag_icon_type',
            [
                'label' => esc_html__('Vedio Icon Type', 'tpcore'),
                'type' => Controls_Manager::CHOOSE,
                'label_block' => false,
                'options' => [
                    'icon' => [
                        'title' => esc_html__('Icon', 'tpcore'),
                        'icon' => 'eicon-nerd-wink',
                    ],
                    'image' => [
                        'title' => esc_html__('Image', 'tpcore'),
                        'icon' => 'fa fa-image',
                    ],
                    'svg' => [
                        'title' => esc_html__('Svg', 'tpcore'),
                        'icon' => 'fas fa-code',
                    ],
                ],
                'default' => 'icon',
                'toggle' => false,
                'style_transfer' => true,
            ]
        );

        $this->add_control(
            'image',
            [
                'label' => esc_html__('Image', 'tpcore'),
                'type' => Controls_Manager::MEDIA,
                'default' => [
                    'url' => Utils::get_placeholder_image_src(),
                ],
                'condition' => [
                    'tp_tag_icon_type' => 'image',
                ],
            ]
        );

        $this->add_control(
            'icon',
            [
                'label' => esc_html__('Icon', 'tpcore'),
                'type' => Controls_Manager::ICONS,
                'default' => [
                    'value' => 'fas fa-play',
                    'library' => 'solid',

                ],
                'condition' => [
                    'tp_tag_icon_type' => 'icon',
                ]
            ]
        );

        $this->add_control(
            'svg',
            [
                'type' => Controls_Manager::TEXTAREA,
                'label_block' => true,
                'label' => __('Svg Code', 'tpcore'),
                'default' => __('Svg Code Here', 'tpcore'),
                'placeholder' => __('Type Svg Code here', 'tpcore'),
                'dynamic' => [
                    'active' => true,
                ],
                'condition' => [
                    'tp_tag_icon_type' => 'svg',
                ],
            ]
        );

        $this->add_control(
            'tp_vedio_link',
            [
                'label' => esc_html__('Vedio Link', 'tpcore'),
                'type' => Controls_Manager::TEXT,
                'default' => esc_html__('#', 'tpcore'),
                'label_block' => true,
            ]
        );
        $this->end_controls_section();


        // image
        $this->start_controls_section(
            'tp_image_sec',
            [
                'label' => esc_html__('Image Section', 'tp-core'),
                'condition' => [
                    'tp_design_style' => ['layout-1']
                ]
            ]
        );

        $this->add_control(
            'tp_shape_image',
            [
                'label' => esc_html__('Shape Image', 'tp-core'),
                'type' => Controls_Manager::MEDIA,
                'default' => [
                    'url' => \Elementor\Utils::get_placeholder_image_src(),
                ],
            ]
        );

        $this->add_control(
            'tp_shape_image2',
            [
                'label' => esc_html__('Shape Image 2', 'tp-core'),
                'type' => Controls_Manager::MEDIA,
                'default' => [
                    'url' => \Elementor\Utils::get_placeholder_image_src(),
                ],
            ]
        );

        $this->add_control(
            'tp_shape_svg_code',
            [
                'label' => esc_html__('Shape SVG Code', 'tp-core'),
                'type' => Controls_Manager::TEXTAREA,
                'rows' => 10,
                'default' => esc_html__('Default Shape SVG Code', 'tp-core'),
                'placeholder' => esc_html__('Type your Shape SVG Code here', 'tp-core'),
            ]
        );

        $this->end_controls_section();

        // additional info
        $this->start_controls_section(
            'tp_add_info_sec',
            [
                'label' => esc_html__('Additional Info', 'tp-core'),
                'condition' => [
                    'tp_design_style' => ['layout-1']
                ]
            ]
        );

        $this->add_control(
            'tp_shape_show',
            [
                'label' => esc_html__('Shape Image Show', 'tpcore'),
                'type' => Controls_Manager::SWITCHER,
                'label_on' => esc_html__('Show', 'tpcore'),
                'label_off' => esc_html__('Hide', 'tpcore'),
                'return_value' => 'yes',
                'default' => 'yes',
                'condition' => [
                    'tp_design_style' => ['layout-1']
                ]
            ]
        );

        $this->add_control(
            'tp_slider_arrow',
            [
                'label' => esc_html__('Next/Prev Arrow Show', 'tpcore'),
                'type' => Controls_Manager::SWITCHER,
                'label_on' => esc_html__('Show', 'tpcore'),
                'label_off' => esc_html__('Hide', 'tpcore'),
                'return_value' => 'yes',
                'default' => 'yes',
            ]
        );
        $this->end_controls_section();

    }

    protected function style_tab_content()
    {
        $this->tp_section_style_controls('testimonial', 'Section - Style', '.tp-el-section');
        $this->tp_icon_style(null, 'testimonial_icon', '.tp-el-v-icon', 'Video Icon/Image/SVG');
        $this->tp_basic_style_controls('testimonial_video_t', 'Video Title', '.tp-el-v-title');
        $this->tp_basic_style_controls('testimonial_testi_h_subtitle', 'Testimonial Heading Subtitle', '.tp-el-testi-h-subtitle');
        $this->tp_basic_style_controls('testimonial_testi_h_title', 'Testimonial Heading Title', '.tp-el-testi-h-subtitle');
        $this->tp_icon_style(null, 'testimonial_quote_icon', '.tp-el-testi-quote', 'Quote - Icon/Image/SVG');
        $this->tp_icon_style(null, 'testimonial_avator', '.tp-el-testi-avator', 'Avator - Icon/Image/SVG');
        $this->tp_basic_style_controls('testimonial_testi_content', 'Testimonial Content', '.tp-el-testi-content');
        $this->tp_basic_style_controls('testimonial_testi_name', 'Testimonial Name', '.tp-el-testi-name');
        $this->tp_basic_style_controls('testimonial_testi_designation', 'Testimonial Designation', '.tp-el-testi-designation');

        $this->start_controls_section(
            'tp_blog_post_avator_section',
                [
                  'label' => esc_html__( 'Avator Image', 'tpcore' ),
                  'tab'   => \Elementor\Controls_Manager::TAB_STYLE,
                ]
           );
           
           $this->add_responsive_control(
               'tp_blog_post_avator_w',
               [
                   'type' => \Elementor\Controls_Manager::SLIDER,
                   'label' => esc_html__('Image Width', 'tpcore'),
                   'size_units' => ['px', '%'],
                   'range' => [
                       'px' => [
                           'min' => 1,
                           'max' => 2000,
                       ],
                   ],
                   'selectors' => [
                       '{{WRAPPER}} .tp-el-avator img' => 'min-width: {{SIZE}}{{UNIT}};',
                   ],
               ]
           );
           $this->add_responsive_control(
               'tp_blog_post_avator_h',
               [
                   'type' => \Elementor\Controls_Manager::SLIDER,
                   'label' => esc_html__('Image Height', 'tpcore'),
                   'size_units' => ['px', '%'],
                   'range' => [
                       'px' => [
                           'min' => 1,
                           'max' => 2000,
                       ],
                   ],
                   'selectors' => [
                       '{{WRAPPER}} .tp-el-avator img' => 'min-height: {{SIZE}}{{UNIT}};',
                   ],
               ]
           );
           
           $this->end_controls_section();


        
    }

    /**
     * Render the widget output on the frontend.
     *
     * Written in PHP and used to generate the final HTML.
     *
     * @since 1.0.0
     *
     * @access protected
     */
    protected function render()
    {
        $settings = $this->get_settings_for_display();
        ?>

        <?php if ($settings['tp_design_style'] == 'layout-2'): ?>

            <div class="row justify-content-center">
                <div class="col-xl-5 col-lg-6 col-md-8">
                    <div class="tp-testimonial-2-avatar-active">
                        <?php foreach ($settings['tp_testimonial_slides'] as $key => $item):

                            $logo = tp_get_img($item, 'testimonial_logo', 'full', false);
                            ?>
                            <div class="tp-testimonial-2-avatar-item tp-el-avator">
                                <img src="<?php echo esc_url($logo['testimonial_logo']); ?>"
                                    alt="<?php echo esc_attr($logo['testimonial_logo_alt']); ?>">
                            </div>
                        <?php endforeach; ?>
                    </div>
                </div>
            </div>
            <div class="row justify-content-center">
                <div class="col-lg-8">
                    <div class="tp-testimonial-2-active">
                        <?php foreach ($settings['tp_testimonial_slides'] as $key => $item):
                            $testimonial_shape_img = tp_get_img($item, 'testimonial_shape_img', 'full', false);
                            $testimonial_shape_img2 = tp_get_img($item, 'testimonial_shape_img2', 'full', false);
                            ?>
                            <div class="tp-testimonial-2-item text-center">
                                <div class="tp-testimonial-2-avatar-info">
                                    <?php if (!empty($item['testimonial_name'])): ?>
                                        <h4 class="tp-testimonial-2-avatar-title tp-el-testi-name">
                                            <?php echo tp_kses($item['testimonial_name']); ?>
                                        </h4>
                                    <?php endif; ?>
                                    <?php if (!empty($item['testimonial_position'])): ?>
                                        <span class="tp-el-testi-designation">
                                            <?php echo tp_kses($item['testimonial_position']); ?>
                                        </span>
                                    <?php endif; ?>
                                    <?php if (!empty($item['tp_testi_rating'])): ?>
                                        <div class="tp-testimonial-2-avatar-rating tp-el-rep-star">
                                            <?php
                                            $tp_rating = $item['tp_testi_rating'];
                                            $tp_rating_minus = 5 - $item['tp_testi_rating'];
                                            for ($i = 1; $i <= $tp_rating; $i++):
                                                ?>
                                                <i class="fa-solid fa-star"></i>
                                            <?php endfor; ?>
                                            <?php
                                            for ($i = 1; $i <= $tp_rating_minus; $i++):
                                                ?>
                                                <i class="fa-light fa-star-sharp"></i>
                                            <?php endfor; ?>
                                        </div>
                                    <?php endif; ?>
                                </div>
                                <div class="tp-testimonial-2-content p-relative">
                                    <?php if (!empty($item['testimonial_content'])): ?>
                                        <p class="tp-el-testi-content">
                                            <?php echo tp_kses($item['testimonial_content']); ?>
                                        </p>
                                    <?php endif; ?>
                                    <div class="tp-testimonial-2-shape">
                                        <div class="shape-1">
                                            <img src="<?php echo esc_url($testimonial_shape_img['testimonial_shape_img']); ?>"
                                                alt="<?php echo esc_attr($testimonial_shape_img['testimonial_shape_img_alt']); ?>">
                                        </div>

                                        <div class="shape-2">
                                            <img src="<?php echo esc_url($testimonial_shape_img2['testimonial_shape_img2']); ?>"
                                                alt="<?php echo esc_attr($testimonial_shape_img2['testimonial_shape_img2_alt']); ?>">
                                        </div>
                                    </div>
                                </div>
                            </div>
                        <?php endforeach; ?>
                    </div>
                </div>
            </div>

        <?php else:
            if (!empty($settings['tp_shape_image']['url'])) {
                $tp_shape_image_url = !empty($settings['tp_shape_image']['id']) ? wp_get_attachment_image_url($settings['tp_shape_image']['id'], 'full') . '' : $settings['tp_shape_image']['url'];
                $tp_shape_image_alt = get_post_meta($settings["tp_shape_image"]["id"], "_wp_attachment_image_alt", true);
            }
            if (!empty($settings['tp_shape_image2']['url'])) {
                $tp_shape_image2_url = !empty($settings['tp_shape_image2']['id']) ? wp_get_attachment_image_url($settings['tp_shape_image2']['id'], 'full') . '' : $settings['tp_shape_image2']['url'];
                $tp_shape_image2_alt = get_post_meta($settings["tp_shape_image2"]["id"], "_wp_attachment_image_alt", true);
            }
            ?>

            <section class="testimonial-area tp-testimonial-bg fix p-relative tp-el-section">
                <?php if (!empty($settings['tp_shape_svg_code'])): ?>
                    <div class="tp-testimonial-themebg">
                        <span></span>
                        <div class="tp-testimonial-themebg-shape">
                            <?php echo tp_kses($settings['tp_shape_svg_code']); ?>
                        </div>
                    </div>
                <?php endif; ?>

                <div class="container">
                    <div class="row align-items-center">
                        <div class="col-lg-6">
                            <div class="tp-video-wrap d-flex">
                                <div class="tp-video-icon text-center">
                                    <a class="video-border-animation popup-video tp-el-v-icon"
                                        href="<?php echo esc_url($settings['tp_vedio_link']); ?>">
                                        <span>
                                            <?php if ($settings['tp_tag_icon_type'] === 'image' && ($settings['image']['url'] || $settings['image']['id'])):
                                                $this->get_render_attribute_string('image');
                                                $settings['hover_animation'] = 'disable-animation';
                                                ?>
                                                <?php echo Group_Control_Image_Size::get_attachment_image_html($settings, 'image'); ?>
                                            <?php elseif (!empty($settings['icon'])): ?>
                                                <?php \Elementor\Icons_Manager::render_icon($settings['icon'], ['aria-hidden' => 'true']); ?>
                                            <?php elseif (!empty($settings['svg'])): ?>
                                                <?php echo $settings['svg']; ?>
                                            <?php endif; ?>
                                        </span>
                                    </a>
                                    <?php if (!empty($settings['tp_vedio_title'])): ?>
                                        <p class="tp-el-v-title"><?php echo tp_kses($settings['tp_vedio_title']); ?></p>
                                    <?php endif; ?>
                                </div>
                            </div>
                        </div>
                        <div class="col-lg-6">
                            <div class="tp-testimonial-wrapper wow fadeInRight" data-wow-delay=".3s">
                                <div class="tp-section mb-40">
                                    <?php if (!empty($settings['tp_subtitle'])): ?>
                                        <h5 class="tp-section-subtitle tp-el-testi-h-subtitle">
                                            <?php echo tp_kses($settings['tp_subtitle']); ?>
                                        </h5>
                                    <?php endif; ?>

                                    <?php if (!empty($settings['tp_title'])): ?>
                                        <h3 class="tp-section-title tp-el-testi-h-title">
                                            <?php echo tp_kses($settings['tp_title']); ?>
                                        </h3>
                                    <?php endif; ?>
                                </div>

                                <div class="tp-testimonial-shape">
                                    <?php if (
                                        ($settings['tp_box_icon_type'] == 'icon' && (!empty($settings['tp_box_icon']) || !empty($settings['tp_box_selected_icon']['value']))) ||
                                        ($settings['tp_box_icon_type'] == 'image' && !empty($settings['tp_box_icon_image']['url'])) ||
                                        ($settings['tp_box_icon_type'] == 'svg' && !empty($settings['tp_box_icon_svg']))
                                    ): ?>
                                        <div class="tp-testimonial-shape-1 tp-el-testi-quote">
                                            <?php $this->tp_icon_show($settings); ?>
                                        </div>
                                    <?php endif; ?>

                                    <?php if (!empty($settings['tp_shape_show'])): ?>
                                        <div class="tp-testimonial-shape-2">
                                            <img src="<?php echo esc_url($tp_shape_image_url) ?>"
                                                alt="<?php echo esc_attr($tp_shape_image_alt) ?>">
                                        </div>
                                        <?php if (!empty($settings['tp_shape_image2']['url'])): ?>
                                            <div class="tp-testimonial-shape-3 wow bounceIn" data-wow-duration="1.5s" data-wow-delay=".4s">
                                                <img src="<?php echo esc_url($tp_shape_image2_url) ?>"
                                                    alt="<?php echo esc_attr($tp_shape_image2_alt) ?>">
                                            </div>
                                        <?php endif; ?>
                                    <?php endif; ?>
                                </div>

                                <div class="swiper tp-testimonial-active">
                                    <div class="swiper-wrapper">
                                        <?php foreach ($settings['tp_testimonial_slides'] as $key => $item):

                                            $logo = tp_get_img($item, 'testimonial_logo', 'full', false);
                                            ?>
                                            <div class="swiper-slide tp-testimonial-item">
                                                <div class="tp-testimonial-avatar tp-el-testi-avator">
                                                    <img src="<?php echo esc_url($logo['testimonial_logo']); ?>"
                                                        alt="<?php echo esc_attr($logo['testimonial_logo_alt']); ?>">
                                                </div>

                                                <?php if (!empty($item['testimonial_content'])): ?>
                                                    <div class="tp-testimonial-content">
                                                        <p class="tp-el-testi-content">
                                                            <?php echo tp_kses($item['testimonial_content']); ?>
                                                        </p>
                                                    </div>
                                                <?php endif; ?>

                                                <div class="tp-testimonial-avatar-info">
                                                    <?php if (!empty($item['testimonial_name'])): ?>
                                                        <h4 class="tp-testimonial-avatar-title tp-el-testi-name">
                                                            <?php echo tp_kses($item['testimonial_name']); ?>
                                                        </h4>
                                                    <?php endif; ?>

                                                    <?php if (!empty($item['testimonial_position'])): ?>
                                                        <span class="tp-el-testi-designation">
                                                            <?php echo tp_kses($item['testimonial_position']); ?>
                                                        </span>
                                                    <?php endif; ?>
                                                </div>
                                            </div>
                                        <?php endforeach; ?>
                                    </div>
                                </div>

                                <?php if (!empty($settings['tp_slider_arrow'])): ?>
                                    <div class="tp-testimonial-arrow text-md-end">
                                        <div class="tp-testimonial-prev">
                                            <span>
                                                <svg width="12" height="12" viewBox="0 0 12 12" fill="none"
                                                    xmlns="http://www.w3.org/2000/svg">
                                                    <path d="M11 6H1" stroke="currentColor" stroke-width="1.5" stroke-linecap="round"
                                                        stroke-linejoin="round" />
                                                    <path d="M6 1L1 6L6 11" stroke="currentColor" stroke-width="1.5"
                                                        stroke-linecap="round" stroke-linejoin="round" />
                                                </svg>
                                            </span>
                                        </div>
                                        <div class="tp-testimonial-next">
                                            <span>
                                                <svg width="12" height="12" viewBox="0 0 12 12" fill="none"
                                                    xmlns="http://www.w3.org/2000/svg">
                                                    <path d="M1 6H11" stroke="currentColor" stroke-width="1.5" stroke-linecap="round"
                                                        stroke-linejoin="round" />
                                                    <path d="M6 1L11 6L6 11" stroke="currentColor" stroke-width="1.5"
                                                        stroke-linecap="round" stroke-linejoin="round" />
                                                </svg>
                                            </span>
                                        </div>
                                    </div>
                                <?php endif; ?>
                            </div>
                        </div>
                    </div>
                </div>
            </section>

        <?php endif;
    }
}

$widgets_manager->register(new TP_Testimonial_Slider());