<?php
/**
 * This template can be overridden by copying it to `bili[-child]/bili-core/elementor/widgets/wgl-showcase.php`.
 */
namespace WGL_Extensions\Widgets;

defined('ABSPATH') || exit; // Abort, if called directly.

use Elementor\{
    Plugin,
    Widget_Base,
    Controls_Manager,
    Icons_Manager,
    Utils,
    Repeater,
    Group_Control_Border,
    Group_Control_Box_Shadow,
    Group_Control_Typography,
    Group_Control_Background
};

use WGL_Extensions\{
    Widgets\WGL_Double_Heading,
    WGL_Framework_Global_Variables as WGL_Globals,
    Includes\WGL_Cursor,
    Includes\WGL_Elementor_Helper
};

class WGL_Showcase extends Widget_Base
{
    public function get_name()
    {
        return 'wgl-showcase';
    }

    public function get_title()
    {
        return esc_html__('WGL Showcase', 'bili-core');
    }

    public function get_icon()
    {
        return 'wgl-showcase';
    }

    public function get_script_depends()
    {
        return [
            'wgl-widgets',
        ];
    }


    public function get_categories()
    {
        return ['wgl-modules'];
    }

    protected function register_controls()
    {
        /**
         * CONTENT -> GENERAL
         */

        $this->start_controls_section(
            'content_general',
            ['label' => esc_html__('General', 'bili-core')]
        );

        $this->add_control(
            'showcase_layout',
            [
                'type' => Controls_Manager::SELECT,
                'label' => esc_html__('Layout', 'bili-core'),
                'options' => [
                    'slide' => esc_html__('Slide', 'bili-core'),
                    'fade_bg' => esc_html__('Fade', 'bili-core'),
                    'interactive' => esc_html__('Interactive', 'bili-core'),
                ],
                'default' => 'slide',
            ]
        );

        $repeater = new Repeater();

        $repeater->add_control(
            'link',
            [
                'label' => esc_html__('Link', 'bili-core'),
                'type' => Controls_Manager::URL,
                'label_block' => true,
            ]
        );

        $repeater->add_control(
            'thumbnail',
            [
                'label' => esc_html__('Image', 'bili-core'),
                'type' => Controls_Manager::MEDIA,
                'label_block' => true,
                'default' => ['url' => Utils::get_placeholder_image_src()],
            ]
        );

        $repeater->add_control(
            'title',
            [
                'label' => esc_html__('Title', 'bili-core'),
                'type' => Controls_Manager::TEXT,
                'label_block' => true,
                'default' => esc_html__('Great Work!​', 'bili-core'),
            ]
        );

        $repeater->add_control(
            'subtitle',
            [
                'label' => esc_html__('Subtitle', 'bili-core'),
                'type' => Controls_Manager::TEXT,
	            'default' => '●',
            ]
        );

        $repeater->add_control(
            'category',
            [
                'label' => esc_html__('Category', 'bili-core'),
                'type' => Controls_Manager::TEXT,
	            'default' => esc_html__('TECHNOLOGY', 'bili-core'),
            ]
        );

        $this->add_control(
            'items',
            [
                'label' => esc_html__('Items', 'bili-core'),
                'type' => Controls_Manager::REPEATER,
	            'fields' => $repeater->get_controls(),
	            'default' => [
		            [
			            'title' => esc_html__( 'Link Title 1', 'bili-core' ),
                        'category' => esc_html__( 'TECHNOLOGY', 'bili-core' ),
			            'link_active' => 'yes'
		            ],
		            [ 'title' => esc_html__( 'Link Title 2', 'bili-core' ), 'category' => esc_html__( 'TECHNOLOGY', 'bili-core' ) ],
		            [ 'title' => esc_html__( 'Link Title 3', 'bili-core' ), 'category' => esc_html__( 'TECHNOLOGY', 'bili-core' ) ],
	            ],
                'title_field' => '{{{ title }}}',
            ]
        );

        $this->add_responsive_control(
            'alignment',
            [
                'label' => esc_html__('Title Alignment', 'bili-core'),
                'type' => Controls_Manager::CHOOSE,
                'separator' => 'before',
                'toggle' => false,
                'options' => [
                    'left' => [
                        'title' => esc_html__('Left', 'bili-core'),
                        'icon' => 'eicon-text-align-left',
                    ],
                    'center' => [
                        'title' => esc_html__('Center', 'bili-core'),
                        'icon' => 'eicon-text-align-center',
                    ],
                    'right' => [
                        'title' => esc_html__('Right', 'bili-core'),
                        'icon' => 'eicon-text-align-right',
                    ],
                ],
                'condition' => [
                    'showcase_layout' => ['slide', 'fade_bg'],
                ],
                'prefix_class' => 'a%s',
                'default' => 'left',
            ]
        );

        $this->add_responsive_control(
		    'title_hover_move',
		    [
			    'label' => esc_html__('Move the title to right', 'bili-core'),
			    'type' => Controls_Manager::SLIDER,
			    'size_units' => [ 'px' ],
			    'default' => [ 'size' => '10', 'unit' => 'px' ],
			    'selectors' => [
				    '{{WRAPPER}} .showcase__title:hover .showcase__title-inner, {{WRAPPER}} .showcase__title.active .showcase__title-inner' => 'transform: translateX({{SIZE}}{{UNIT}})',
				    '{{WRAPPER}} .showcase__title .showcase__title-inner' => 'padding-left: {{SIZE}}{{UNIT}}; margin-left: -{{SIZE}}{{UNIT}}',
			    ],
                'condition' => [
                    'showcase_layout' => 'slide',
                ],
		    ]
	    );

        $this->add_control(
			'height_slide',
			[
				'label' => esc_html__( 'Height', 'bili-core' ),
				'type' => Controls_Manager::SELECT,
				'default' => 'min-height',
				'options' => [
					'full' => esc_html__( 'Fit To Screen', 'bili-core' ),
					'min-height' => esc_html__( 'Min Height', 'bili-core' ),
				],
				'prefix_class' => 'wgl-showcase-height-',
                'condition' => [
                    'showcase_layout' => ['slide', 'fade_bg'],
                ]
			]
		);

		$this->add_responsive_control(
			'custom_height_slide',
			[
				'label' => esc_html__( 'Minimum Height', 'bili-core' ),
				'type' => Controls_Manager::SLIDER,
				'default' => [
					'size' => 660,
				],
				'range' => [
					'px' => [ 'min' => 0, 'max' => 2560 ],
					'vw' => [ 'min' => 0, 'max' => 200 ],
					'vh' => [ 'min' => 0, 'max' => 150 ],
				],
				'selectors' => [
					'{{WRAPPER}} .showcase__wrapper' => 'height: {{SIZE}}{{UNIT}};',
				],
				'condition' => [
					'height_slide' => [ 'min-height' ],
				],
				'size_units' => [ 'px', 'vh', 'vw' ],
			]
		);

		$this->add_responsive_control(
			'custom_image_container_width',
			[
				'label' => esc_html__( 'Image Container Width', 'bili-core' ),
				'type' => Controls_Manager::SLIDER,
				'size_units' => [ 'px', '%', 'vw' ],
				'range' => [
					'px' => [ 'min' => 0, 'max' => 2560 ],
					'vw' => [ 'min' => 0, 'max' => 200 ],
					'%' => [ 'min' => 0, 'max' => 200 ],
				],
				'default' => [ 'size' => 100, 'unit' => 'vw' ],
				'selectors' => [
					'{{WRAPPER}} .showcase__images' => 'width: {{SIZE}}{{UNIT}};',
				],
				'condition' => [
					'height_slide' => [ 'min-height' ],
					'showcase_layout' => 'fade_bg'
				],
			]
		);

        $this->add_control(
            'img_size_string',
            [
                'type' => Controls_Manager::SELECT,
                'label' => esc_html__('Image Size', 'bili-core'),
                'separator' => 'before',
                'options' => [
                    '150' => esc_html__('150x150 - Thumbnail', 'bili-core'),
                    '300' => esc_html__('300x300 - Medium', 'bili-core'),
                    '768' => esc_html__('768x768 - Medium Large', 'bili-core'),
                    '1024' => esc_html__('1024x1024 - Large', 'bili-core'),
                    'full' => esc_html__('Full', 'bili-core'),
                    'custom' => esc_html__('Custom', 'bili-core'),
                ],
                'default' => 'full',
            ]
        );

        $this->add_control(
            'img_size_array',
            [
                'label' => esc_html__('Image Dimension', 'bili-core'),
                'type' => Controls_Manager::IMAGE_DIMENSIONS,
                'condition' => [
                    'img_size_string' => 'custom',
                ],
                'description' => esc_html__('Crop the original image to any custom size. You can also set a single value for width to keep the initial ratio.', 'bili-core'),
            ]
        );

        $this->add_control(
            'img_aspect_ratio',
            [
                'label' => esc_html__('Image Aspect Ratio', 'bili-core'),
                'type' => Controls_Manager::SELECT,
                'options' => [
                    '' => esc_html__('No Crop', 'bili-core'),
                    '1:1' => esc_html__('1:1', 'bili-core'),
                    '3:2' => esc_html__('3:2', 'bili-core'),
                    '4:3' => esc_html__('4:3', 'bili-core'),
                    '6:5' => esc_html__('6:5', 'bili-core'),
                    '9:16' => esc_html__('9:16', 'bili-core'),
                    '16:9' => esc_html__('16:9', 'bili-core'),
                    '21:9' => esc_html__('21:9', 'bili-core'),
                ],
                'default' => '1:1',
            ]
        );

        $this->end_controls_section();


        /*-----------------------------------------------------------------------------------*/
		/*  CONTENT -> Cursor
		/*-----------------------------------------------------------------------------------*/

        WGL_Cursor::init(
            $this,
            [
                'section' => true,
            ]
        );

         /**
         * CONTENT -> BUTTON
         */

        $this->start_controls_section(
            'section_style_link',
            [
                'label' => esc_html__('Button', 'bili-core'),
            ]
        );

        $this->add_control(
            'add_item_link',
            [
                'label' => esc_html__('Whole Item Link', 'bili-core'),
                'type' => Controls_Manager::SWITCHER,
                'label_on' => esc_html__('On', 'bili-core'),
                'label_off' => esc_html__('Off', 'bili-core'),
                'default' => 'yes',
                'condition' => [
                    'showcase_layout' => 'interactive',
                ]
            ]
        );

        $this->add_control(
            'add_read_more',
            [
                'label' => esc_html__('Button', 'bili-core'),
                'type' => Controls_Manager::SWITCHER,
                'label_on' => esc_html__('On', 'bili-core'),
                'label_off' => esc_html__('Off', 'bili-core'),
                'default' => 'yes',
            ]
        );

        $this->end_controls_section();


        /**
         * STYLE -> ITEM CONTAINER
         */

        $this->start_controls_section(
            'style_item_container',
            [
                'label' => esc_html__('Item Container', 'bili-core'),
                'tab' => Controls_Manager::TAB_STYLE,
            ]
        );

        $this->add_responsive_control(
            'item_margin',
            [
                'label' => esc_html__('Margin', 'bili-core'),
                'type' => Controls_Manager::DIMENSIONS,
                'size_units' => ['px'],
                'selectors' => [
                    '{{WRAPPER}} .wgl-showcase .showcase__wrapper, {{WRAPPER}} .wgl-showcase .showcase__content' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
                ],
            ]
        );

        $this->add_responsive_control(
            'item_padding',
            [
                'label' => esc_html__('Padding', 'bili-core'),
                'type' => Controls_Manager::DIMENSIONS,
                'size_units' => ['px', 'em', '%'],
                'selectors' => [
                    '{{WRAPPER}} .wgl-showcase .showcase__wrapper, {{WRAPPER}} .wgl-showcase .showcase__content' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->add_control(
            'item_radius',
            [
                'label' => esc_html__('Border Radius', 'bili-core'),
                'type' => Controls_Manager::DIMENSIONS,
                'size_units' => ['px', '%'],
                'selectors' => [
                    '{{WRAPPER}} .wgl-showcase .showcase__wrapper, {{WRAPPER}} .wgl-showcase .showcase__content' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->add_group_control(
            Group_Control_Background::get_type(),
            [
                'name' => 'item_bg',
                'selector' => '{{WRAPPER}} .wgl-showcase .showcase__wrapper, {{WRAPPER}} .wgl-showcase .showcase__content',
            ]
        );

        $this->add_group_control(
            Group_Control_Border::get_type(),
            [
                'name' => 'item_border',
                'selector' => '{{WRAPPER}} .wgl-showcase .showcase__wrapper, {{WRAPPER}} .wgl-showcase .showcase__content',
            ]
        );

        $this->add_group_control(
            Group_Control_Box_Shadow::get_type(),
            [
                'name' => 'item_shadow',
                'selector' => '{{WRAPPER}} .wgl-showcase .showcase__wrapper, {{WRAPPER}} .wgl-showcase .showcase__content',
            ]
        );

        $this->add_responsive_control(
            'image_wrapper_size',
            [
                'label' => esc_html__('Image Cointainer Size', 'bili-core'),
                'type' => Controls_Manager::SLIDER,
                'size_units' => ['%', 'px'],
                'range' => [
                    '%' => ['max' => 100],
                    'px' => ['max' => 1920],
                ],
	            'condition' => ['showcase_layout' => 'slide'],
                'default' => ['size' => 100, 'unit' => '%'],
                'selectors' => [
                    '{{WRAPPER}} .wgl-showcase.slide-showcase .showcase__images' => 'width: {{SIZE}}{{UNIT}};',
                ],
            ]
        );

        $this->add_responsive_control(
            'title_wrapper_size',
            [
                'label' => esc_html__('Title Cointainer Size', 'bili-core'),
                'type' => Controls_Manager::SLIDER,
                'size_units' => ['%'],
                'range' => [
                    '%' => ['max' => 100],
                ],
                'condition' => ['showcase_layout' => 'slide'],
                'default' => ['size' => 100, 'unit' => '%'],
                'selectors' => [
                    '{{WRAPPER}} .wgl-showcase.slide-showcase .showcase__titles' => 'width: {{SIZE}}%;',
                ],
            ]
        );

	    $this->end_controls_section();

        /**
         * STYLE -> IMAGE
         */

        $this->start_controls_section(
            'style_image',
            [
                'label' => esc_html__('Image', 'bili-core'),
                'tab' => Controls_Manager::TAB_STYLE,
            ]
        );

        $this->start_controls_tabs('image');

        $this->start_controls_tab(
            'image_idle',
            ['label' => esc_html__('Idle', 'bili-core')]
        );

        $this->add_responsive_control(
		    'image_max_height',
		    [
			    'label' => esc_html__('Image Max Height', 'bili-core'),
			    'type' => Controls_Manager::SLIDER,
			    'size_units' => ['%', 'px'],
			    'default' => [ 'size' => '88', 'unit' => '%' ],
			    'selectors' => [
				    '{{WRAPPER}} .showcase__images .showcase__image' => 'max-height: {{SIZE}}{{UNIT}}',
			    ],
			    'condition' => ['showcase_layout' => 'slide'],
		    ]
	    );

        $this->add_responsive_control(
            'image_top_offset',
            [
                'label' => esc_html__('Top Offset', 'bili-core'),
                'type' => Controls_Manager::SLIDER,
                'size_units' => ['%', 'px'],
                'range' => [
                    '%' => ['min' => -200, 'max' => 200],
                    'px' => ['min' => -1000, 'max' => 1000, 'step' => 1],
                ],
                'default' => [ 'size' => 75, 'unit' => 'px' ],
	            'condition' => ['showcase_layout' => 'slide'],
                'selectors' => [
                    '{{WRAPPER}} .showcase__images' => 'top: {{SIZE}}{{UNIT}}',
                ],
            ]
        );

        $this->add_responsive_control(
            'image_left_offset',
            [
                'label' => esc_html__('Left Offset', 'bili-core'),
                'type' => Controls_Manager::SLIDER,
                'size_units' => ['%', 'px'],
                'range' => [
                    '%' => ['min' => -200, 'max' => 200],
                    'px' => ['min' => -1000, 'max' => 1000, 'step' => 1],
                ],
                'default' => [ 'size' => 288, 'unit' => 'px' ],
	            'condition' => ['showcase_layout' => 'slide'],
                'selectors' => [
                    '{{WRAPPER}} .showcase__images' => 'left: {{SIZE}}{{UNIT}}',
                ],
            ]
        );

        $this->add_control(
            'image_radius_idle',
            [
                'label' => esc_html__('Border Radius', 'bili-core'),
                'type' => Controls_Manager::DIMENSIONS,
                'size_units' => ['px', '%'],
                'selectors' => [
                    '{{WRAPPER}} .wgl-showcase .showcase__images .showcase__image' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->add_group_control(
            Group_Control_Border::get_type(),
            [
                'name' => 'image_border_idle',
                'selector' => '{{WRAPPER}} .wgl-showcase .showcase__images .showcase__image',
            ]
        );

        $this->add_group_control(
            Group_Control_Box_Shadow::get_type(),
            [
                'name' => 'image_shadow_idle',
                'selector' => '{{WRAPPER}} .wgl-showcase .showcase__images .showcase__image',
            ]
        );

        $this->add_group_control(
            Group_Control_Background::get_type(),
            [
                'name' => 'image_bg_idle',
                'types' => ['classic', 'gradient'],
                'selector' => '{{WRAPPER}} .wgl-showcase .showcase__images .showcase__image:before',
            ]
        );

        $this->end_controls_tab();

        $this->start_controls_tab(
            'image_hover',
            ['label' => esc_html__('Hover', 'bili-core')]
        );

        $this->add_control(
            'image_radius_hover',
            [
                'label' => esc_html__('Border Radius', 'bili-core'),
                'type' => Controls_Manager::DIMENSIONS,
                'size_units' => ['px', '%'],
                'selectors' => [
                    '{{WRAPPER}} .wgl-showcase .showcase__images .showcase__image:hover' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->add_group_control(
            Group_Control_Border::get_type(),
            [
                'name' => 'image_border_hover',
                'selector' => '{{WRAPPER}} .wgl-showcase .showcase__images .showcase__image:hover',
            ]
        );

        $this->add_group_control(
            Group_Control_Box_Shadow::get_type(),
            [
                'name' => 'image_shadow_hover',
                'selector' => '{{WRAPPER}} .wgl-showcase .showcase__images .showcase__image:hover',
            ]
        );

        $this->add_group_control(
            Group_Control_Background::get_type(),
            [
                'name' => 'image_bg_hover',
                'types' => ['classic', 'gradient'],
                'selector' => '{{WRAPPER}} .wgl-showcase .showcase__images .showcase__image:after',
            ]
        );

        $this->end_controls_tab();
        $this->end_controls_tabs();

        $this->add_hidden_device_controls();

        $this->end_controls_section();

        /**
         * STYLE -> Title
         */

        $this->start_controls_section(
            'style_title',
            [
                'label' => esc_html__('Title', 'bili-core'),
                'tab' => Controls_Manager::TAB_STYLE,
            ]
        );

	    $this->add_responsive_control(
		    'title_container_padding',
		    [
			    'label' => esc_html__('Container Padding', 'bili-core'),
			    'type' => Controls_Manager::DIMENSIONS,
			    'size_units' => ['px', 'em', '%'],
			    'condition' => [ 'showcase_layout' => 'fade_bg' ],
			    'separator' => 'before',
			    'selectors' => [
				    '{{WRAPPER}} .wgl-showcase .showcase__titles' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
			    ],
		    ]
	    );

	    $this->add_control(
            'title_tag',
            [
                'label' => esc_html__('HTML tag', 'bili-core'),
                'type' => Controls_Manager::SELECT,
                'options' => [
                    'h1' => esc_html('‹h1›'),
                    'h2' => esc_html('‹h2›'),
                    'h3' => esc_html('‹h3›'),
                    'h4' => esc_html('‹h4›'),
                    'h5' => esc_html('‹h5›'),
                    'h6' => esc_html('‹h6›'),
                    'div' => esc_html('‹div›'),
                    'span' => esc_html('‹span›'),
                ],
                'default' => 'div',
            ]
        );

	    $this->add_group_control(
		    Group_Control_Typography::get_type(),
		    [
			    'name' => 'custom_font_title_1st',
			    'selector' => '{{WRAPPER}} .wgl-showcase .showcase__titles .showcase__title .title,{{WRAPPER}} .wgl-showcase.interactive-showcase .title',
		    ]
	    );

        $this->add_responsive_control(
		    'title_stroke_size',
		    [
			    'label' => esc_html__('Stroke Width', 'bili-core'),
			    'type' => Controls_Manager::SLIDER,
			    'size_units' => ['px'],
			    'range' => ['px' => ['min' => 0, 'max' => 2, 'step' => 0.1]],
			    'selectors' => [
				    '{{WRAPPER}} .wgl-showcase .showcase__titles .showcase__title .title' => '-webkit-text-stroke-width: {{SIZE}}{{UNIT}};',
                    '{{WRAPPER}} .wgl-showcase.interactive-showcase .title' => '-webkit-text-stroke-width: {{SIZE}}{{UNIT}};',
			    ],
		    ]
	    );

	    $this->add_responsive_control(
		    'title_width',
		    [
			    'label' => esc_html__( 'Titles Width', 'bili-core' ),
			    'type' => Controls_Manager::SLIDER,
			    'size_units' => [ 'px', '%', 'vw' ],
			    'range' => [
				    'px' => [ 'min' => 0, 'max' => 500 ],
				    '%' => [ 'min' => 0, 'max' => 100 ],
			    ],
			    'default' => [ 'size' => 33.3, 'unit' => '%' ],
			    'condition' => [ 'showcase_layout' => 'fade_bg' ],
			    'selectors' => [
				    '{{WRAPPER}} .showcase__title' => 'width: {{SIZE}}{{UNIT}};',
			    ],
		    ]
	    );

	    $this->add_responsive_control(
		    'titles_gap',
		    [
			    'label' => esc_html__( 'Titles Gap', 'bili-core' ),
			    'type' => Controls_Manager::SLIDER,
			    'size_units' => [ 'px', '%', 'vw' ],
			    'range' => [
				    'px' => [ 'min' => 0, 'max' => 70, 'step' => 2 ],
				    '%' => [ 'min' => 0, 'max' => 20 ],
			    ],
			    'default' => [ 'size' => 30, 'unit' => 'px' ],
			    'mobile_default' => ['size' => 0, 'unit' => 'px'],
			    'condition' => [ 'showcase_layout' => 'fade_bg' ],
			    'selectors' => [
				    '{{WRAPPER}} .showcase__titles' => '--wgl-showcase-titles-gap: {{SIZE}}{{UNIT}};',
			    ],
		    ]
	    );

	    $this->add_control(
		    'title_v_position',
		    [
			    'label' => esc_html__( 'Title Vertical Position', 'bili-core' ),
			    'type' => Controls_Manager::CHOOSE,
			    'default' => 'align-content: flex-end; align-items: flex-end;',
			    'condition' => [ 'showcase_layout' => 'fade_bg' ],
			    'options' => [
				    'align-content: flex-start; align-items: flex-start;' => [
					    'title' => esc_html__( 'Top', 'bili-core' ),
					    'icon' => 'eicon-justify-start-v',
				    ],
				    'align-content: center; align-items: center;' => [
					    'title' => esc_html__( 'Middle', 'bili-core' ),
					    'icon' => 'eicon-justify-center-v',
				    ],
				    'align-content: flex-end; align-items: flex-end;' => [
					    'title' => esc_html__( 'Bottom', 'bili-core' ),
					    'icon' => 'eicon-justify-end-v',
				    ],
			    ],
			    'selectors' => [
				    '{{WRAPPER}} .showcase__titles' => '{{VALUE}};',
			    ],
		    ]
	    );

        $this->add_responsive_control(
            'info_padding',
            [
                'label' => esc_html__('Padding', 'bili-core'),
                'type' => Controls_Manager::DIMENSIONS,
                'size_units' => ['px', 'em', '%'],
                'selectors' => [
                    '{{WRAPPER}} .wgl-showcase .showcase__titles .showcase__title-inner,
                     {{WRAPPER}} .wgl-showcase.interactive-showcase .title' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->add_responsive_control(
            'info_margin',
            [
                'label' => esc_html__('Margin', 'bili-core'),
                'type' => Controls_Manager::DIMENSIONS,
                'size_units' => ['px', 'em', '%'],
                'selectors' => [
                    '{{WRAPPER}} .wgl-showcase .showcase__titles .showcase__title,
                     {{WRAPPER}} .wgl-showcase.interactive-showcase .title' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

	    $this->start_controls_tabs('title_bg_tabs');

        $this->start_controls_tab(
            'title_bg_tab',
            ['label' => esc_html__('Idle', 'bili-core')]
        );

	    $this->add_control(
		    'title_1st_color',
		    [
			    'label' => esc_html__('Color', 'bili-core'),
			    'type' => Controls_Manager::COLOR,
			    'dynamic' => ['active' => true],
			    'selectors' => [
				    '{{WRAPPER}} .wgl-showcase .showcase__titles .showcase__title-inner,
				     {{WRAPPER}} .wgl-showcase.interactive-showcase .title' => 'color: {{VALUE}};'
			    ],
		    ]
	    );

        $this->add_control(
            'title_1st_strok_idle',
            [
                'label' => esc_html__('Stroke Color', 'bili-core'),
                'type' => Controls_Manager::COLOR,
                'condition' => [
                	'title_stroke_size[size]!' => '',
                ],
                'dynamic' => ['active' => true],
                'selectors' => [
				    '{{WRAPPER}} .wgl-showcase .showcase__titles .showcase__title-inner,
				     {{WRAPPER}} .wgl-showcase.interactive-showcase .title' => '-webkit-text-stroke-color: {{VALUE}};'
			    ],
            ]
        );

        $this->add_control(
            'title_bg',
            [
                'label' => esc_html__('Bakground Color', 'bili-core'),
                'type' => Controls_Manager::COLOR,
                'dynamic' => ['active' => true],
                'selectors' => [
                    '{{WRAPPER}} .wgl-showcase .showcase__titles .showcase__title-inner,
                     {{WRAPPER}} .wgl-showcase.interactive-showcase .title' => 'background-color: {{VALUE}};',
                ],
            ]
        );


	    $this->add_group_control(
		    Group_Control_Box_Shadow::get_type(),
		    [
			    'name' => 'titles_shadow_idle',
			    'condition' => [ 'showcase_layout' => 'fade_bg' ],
			    'fields_options' => [
				    'box_shadow_type' => [
					    'default' => 'yes'
				    ],
				    'box_shadow' => [
					    'default' => [
						    'horizontal' => 0,
						    'vertical' => 1,
						    'blur' => 0,
						    'spread' => 0,
						    'color' => 'rgba(255,255,255,0.5)',
					    ]
				    ]
			    ],
			    'selector' => '{{WRAPPER}} .showcase__titles .showcase__title-inner',
		    ]
	    );

	    $this->end_controls_tab();

        $this->start_controls_tab(
            'title_bg_hover_tab',
            ['label' => esc_html__('Hover/Active', 'bili-core')]
        );

	    $this->add_control(
		    'title_1st_hover',
		    [
			    'label' => esc_html__('Color', 'bili-core'),
			    'type' => Controls_Manager::COLOR,
			    'dynamic' => ['active' => true],
			    'selectors' => [
				    '{{WRAPPER}} .wgl-showcase .showcase__titles .showcase__title.active .showcase__title-inner,
                    {{WRAPPER}} .wgl-showcase.interactive-showcase .title:hover,
                    {{WRAPPER}} .wgl-showcase.interactive-showcase .showcase__link:hover ~ .showcase__title .title' => 'color: {{VALUE}};'
			    ],
		    ]
	    );

        $this->add_control(
            'title_1st_stroke_hover',
            [
                'label' => esc_html__('Stroke Color', 'bili-core'),
                'type' => Controls_Manager::COLOR,
                'condition' => [
                	'title_stroke_size[size]!' => '',
                ],
                'dynamic' => ['active' => true],
                'selectors' => [
				    '{{WRAPPER}} .wgl-showcase .showcase__titles .showcase__title.active .showcase__title-inner,
                    {{WRAPPER}} .wgl-showcase.interactive-showcase .title:hover,
                    {{WRAPPER}} .wgl-showcase.interactive-showcase .showcase__link:hover ~ .showcase__title .title' => '-webkit-text-stroke-color: {{VALUE}};'
			    ],
            ]
        );

        $this->add_control(
            'title_bg_hover',
            [
                'label' => esc_html__('Background Color', 'bili-core'),
                'type' => Controls_Manager::COLOR,
                'dynamic' => ['active' => true],
                'selectors' => [
                    '{{WRAPPER}} .wgl-showcase .showcase__titles .showcase__title:hover .showcase__title-inner,
                    {{WRAPPER}} .wgl-showcase .showcase__titles .showcase__title.active .showcase__title-inner,
                    {{WRAPPER}} .wgl-showcase.interactive-showcase .title:hover,
                    {{WRAPPER}} .wgl-showcase.interactive-showcase .showcase__link:hover ~ .showcase__title .title' => 'background-color: {{VALUE}};',
                ],
            ]
        );

	    $this->add_group_control(
		    Group_Control_Box_Shadow::get_type(),
		    [
			    'name' => 'titles_shadow_hover',
			    'condition' => [ 'showcase_layout' => 'fade_bg' ],
			    'fields_options' => [
				    'box_shadow_type' => [
					    'default' => 'yes'
				    ],
				    'box_shadow' => [
					    'default' => [
						    'horizontal' => 0,
						    'vertical' => 3,
						    'blur' => 0,
						    'spread' => 0,
						    'color' => 'rgba(255,255,255,1)',
					    ]
				    ]
			    ],
			    'selector' => '{{WRAPPER}} .showcase__titles .showcase__title:hover .showcase__title-inner,
			     {{WRAPPER}} .showcase__titles .showcase__title.active .showcase__title-inner',
		    ]
	    );

	    $this->end_controls_tab();
        $this->end_controls_tabs();
        $this->end_controls_section();


        /**
         * STYLE -> Subtitle
         */

        $this->start_controls_section(
            'style_subtitle',
            [
                'label' => esc_html__('Subtitle', 'bili-core'),
                'tab' => Controls_Manager::TAB_STYLE,
            ]
        );

	    $this->add_control(
		    'subtitle_display',
		    [
			    'label' => esc_html__( 'Display', 'bili-core' ),
			    'type' => Controls_Manager::SELECT,
			    'options' => [
				    'inline-block; width: auto;' => esc_html__('inline-block', 'bili-core'),
				    'block;' => esc_html__('block', 'bili-core'),
			    ],
			    'default' => 'inline-block; width: auto;',
			    'selectors' => [
				    '{{WRAPPER}} .showcase__titles .showcase__title .subtitle' => 'display: {{VALUE}};',
			    ],
		    ]
	    );

        $this->add_responsive_control(
            'subtitle_min_width',
            [
                'label' => esc_html__('Min Width', 'bili-core'),
                'type' => Controls_Manager::SLIDER,
                'size_units' => ['px','em','%','vw'],
                'condition' => [ 'subtitle_display' => 'inline-block; width: auto;' ],
                'range' => [
                    'px' => [
                        'min' => 20,
                        'max' => 400,
                    ],
                    '%' => [
                        'min' => 1,
                        'max' => 10,
                    ],
                    'vw' => [
                        'min' => 1,
                        'max' => 10,
                    ],
                    'em' => [
                        'min' => 1,
                        'max' => 10,
                        'step' => 0.2,
                    ],
                ],
                'selectors' => [
                    '{{WRAPPER}} .wgl-showcase .subtitle' => 'min-width: {{SIZE}}{{UNIT}};',
                ],
            ]
        );

	    $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name' => 'custom_font_subtitle',
                'selector' => '{{WRAPPER}} .wgl-showcase .showcase__titles .showcase__title .subtitle, {{WRAPPER}} .wgl-showcase.interactive-showcase .subtitle',
            ]
        );

        $this->add_responsive_control(
		    'subtitle_stroke_size',
		    [
			    'label' => esc_html__('Stroke Width', 'bili-core'),
			    'type' => Controls_Manager::SLIDER,
			    'size_units' => ['px'],
			    'range' => ['px' => ['min' => 0, 'max' => 2, 'step' => 0.1]],
                'default' => ['size' => 1, 'units' => 'px'],
			    'selectors' => [
				    '{{WRAPPER}} .wgl-showcase .showcase__titles .showcase__title .subtitle' => '-webkit-text-stroke-width: {{SIZE}}{{UNIT}};',
                    '{{WRAPPER}} .wgl-showcase.interactive-showcase .subtitle' => '-webkit-text-stroke-width: {{SIZE}}{{UNIT}};',
			    ],
		    ]
	    );

        $this->add_responsive_control(
            'subtitle_margin',
            [
                'label' => esc_html__('Margin', 'bili-core'),
                'type' => Controls_Manager::DIMENSIONS,
                'size_units' => ['px', '%'],
                'selectors' => [
                    '{{WRAPPER}} .subtitle' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->start_controls_tabs( 'subtitle_tabs' );

        $this->start_controls_tab(
            'subtitle_tab',
            ['label' => esc_html__('Idle' , 'bili-core')]
        );

        $this->add_control(
            'subtitle_color',
            [
                'label' => esc_html__('Color', 'bili-core'),
				'type' => Controls_Manager::COLOR,
				'dynamic' => ['active' => true],
                'selectors' => [
                    '{{WRAPPER}} .wgl-showcase .showcase__titles .showcase__title .subtitle,  {{WRAPPER}} .wgl-showcase.interactive-showcase .subtitle' => 'color: {{VALUE}};'
                ],
            ]
        );

        $this->add_control(
            'subtitle_color_idle',
            [
                'label' => esc_html__('Stroke Color', 'bili-core'),
                'type' => Controls_Manager::COLOR,
                'condition' => [
                	'subtitle_stroke_size[size]!' => '',
                ],
                'dynamic' => ['active' => true],
                'selectors' => [
                    '{{WRAPPER}} .wgl-showcase .showcase__titles .showcase__title .subtitle,  {{WRAPPER}} .wgl-showcase.interactive-showcase .subtitle' => '-webkit-text-stroke-color: {{VALUE}};',
                ],
            ]
        );

        $this->end_controls_tab();

        $this->start_controls_tab(
            'subtitle_hover_tab',
            ['label' => esc_html__('Hover/Active' , 'bili-core')]
        );

        $this->add_control(
            'subtitle_hover',
            [
                'label' => esc_html__('Color', 'bili-core'),
				'type' => Controls_Manager::COLOR,
				'dynamic' => ['active' => true],
                'selectors' => [
                    '{{WRAPPER}} .wgl-showcase .showcase__titles .showcase__title.active .subtitle,
                    {{WRAPPER}} .wgl-showcase.interactive-showcase .subtitle:hover,
                    {{WRAPPER}} .wgl-showcase.interactive-showcase .showcase__link:hover ~ .showcase__title .subtitle' => 'color: {{VALUE}};'
                ],
            ]
        );

        $this->add_control(
            'subtitle_color_hover',
            [
                'label' => esc_html__('Stroke Color', 'bili-core'),
                'type' => Controls_Manager::COLOR,
                'condition' => [
                	'subtitle_stroke_size[size]!' => '',
                ],
                'dynamic' => ['active' => true],
                'selectors' => [
                    '{{WRAPPER}} .wgl-showcase .showcase__titles .showcase__title.active .subtitle,
                    {{WRAPPER}} .wgl-showcase.interactive-showcase .subtitle:hover,
                    {{WRAPPER}} .wgl-showcase.interactive-showcase .showcase__link:hover ~ .showcase__title .subtitle' => '-webkit-text-stroke-color: {{VALUE}};'
                ],
            ]
        );

        $this->end_controls_tab();
        $this->end_controls_tabs();

        $this->end_controls_section();

        /**
         * STYLE -> Category
         */

        $this->start_controls_section(
            'style_category',
            [
                'label' => esc_html__('Category', 'bili-core'),
                'tab' => Controls_Manager::TAB_STYLE,
            ]
        );


	    $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name' => 'custom_font_category',
                'selector' => '{{WRAPPER}} .wgl-showcase .showcase__titles .showcase__title .category, {{WRAPPER}} .wgl-showcase.interactive-showcase .category',
            ]
        );

        $this->add_responsive_control(
		    'category_stroke_size',
		    [
			    'label' => esc_html__('Stroke Width', 'bili-core'),
			    'type' => Controls_Manager::SLIDER,
			    'size_units' => ['px'],
			    'range' => ['px' => ['min' => 0, 'max' => 2, 'step' => 0.1]],
			    'selectors' => [
				    '{{WRAPPER}} .wgl-showcase .showcase__titles .showcase__title .category' => '-webkit-text-stroke-width: {{SIZE}}{{UNIT}};',
                    '{{WRAPPER}} .wgl-showcase.interactive-showcase .category' => '-webkit-text-stroke-width: {{SIZE}}{{UNIT}};',
			    ],
		    ]
	    );

        $this->add_responsive_control(
            'category_margin',
            [
                'label' => esc_html__('Margin', 'bili-core'),
                'type' => Controls_Manager::DIMENSIONS,
                'size_units' => ['px', '%'],
                'selectors' => [
                    '{{WRAPPER}} .category' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->start_controls_tabs( 'category_tabs' );

        $this->start_controls_tab(
            'category_tab',
            ['label' => esc_html__('Idle' , 'bili-core')]
        );

        $this->add_control(
            'category_color',
            [
                'label' => esc_html__('Color', 'bili-core'),
				'type' => Controls_Manager::COLOR,
				'dynamic' => ['active' => true],
                'selectors' => [
                    '{{WRAPPER}} .wgl-showcase .showcase__titles .showcase__title .category,  {{WRAPPER}} .wgl-showcase.interactive-showcase .category' => 'color: {{VALUE}};'
                ],
            ]
        );

        $this->add_control(
            'category_color_idle',
            [
                'label' => esc_html__('Stroke Color', 'bili-core'),
                'type' => Controls_Manager::COLOR,
                'condition' => [
                	'category_stroke_size[size]!' => '',
                ],
                'dynamic' => ['active' => true],
                'selectors' => [
                    '{{WRAPPER}} .wgl-showcase .showcase__titles .showcase__title .category,  {{WRAPPER}} .wgl-showcase.interactive-showcase .category' => '-webkit-text-stroke-color: {{VALUE}};',
                ],
            ]
        );

        $this->end_controls_tab();

        $this->start_controls_tab(
            'category_hover_tab',
            ['label' => esc_html__('Hover/Active' , 'bili-core')]
        );

        $this->add_control(
            'category_hover',
            [
                'label' => esc_html__('Color', 'bili-core'),
				'type' => Controls_Manager::COLOR,
				'dynamic' => ['active' => true],
                'selectors' => [
                    '{{WRAPPER}} .wgl-showcase .showcase__titles .showcase__title.active .category,
                    {{WRAPPER}} .wgl-showcase.interactive-showcase .category:hover,
                    {{WRAPPER}} .wgl-showcase.interactive-showcase .showcase__link:hover ~ .showcase__title .category' => 'color: {{VALUE}};'
                ],
            ]
        );

        $this->add_control(
            'category_color_hover',
            [
                'label' => esc_html__('Stroke Color', 'bili-core'),
                'type' => Controls_Manager::COLOR,
                'condition' => [
                	'category_stroke_size[size]!' => '',
                ],
                'dynamic' => ['active' => true],
                'selectors' => [
                    '{{WRAPPER}} .wgl-showcase .showcase__titles .showcase__title.active .category,
                    {{WRAPPER}} .wgl-showcase.interactive-showcase .category:hover,
                    {{WRAPPER}} .wgl-showcase.interactive-showcase .showcase__link:hover ~ .showcase__title .category' => '-webkit-text-stroke-color: {{VALUE}};'
                ],
            ]
        );

        $this->end_controls_tab();
        $this->end_controls_tabs();

        $this->end_controls_section();

        /**
		 * STYLE -> BUTTON
		 */
        $this->start_controls_section(
            'button_style_section',
            [
                'label' => esc_html__('Button', 'bili-core'),
                'tab' => Controls_Manager::TAB_STYLE,
            ]
        );

        $this->add_responsive_control(
            'button_margin',
            [
                'label' => esc_html__('Margin', 'bili-core'),
                'type' => Controls_Manager::DIMENSIONS,
                'size_units' => ['px', '%'],
                'selectors' => [
                    '{{WRAPPER}} .showcase__button' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->start_controls_tabs('button');

        $this->start_controls_tab(
            'tab_button_idle',
            ['label' => esc_html__('Idle' , 'bili-core') ]
        );

        $this->add_control(
            'button_color_idle',
            [
                'label' => esc_html__('Color', 'bili-core'),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .wgl-showcase_button.icon-read-more i, {{WRAPPER}} .wgl-showcase_button.icon-read-more span' => 'color: {{VALUE}};',
                ],
            ]
        );

        $this->end_controls_tab();

	    $this->start_controls_tab(
		    'tab_button_hover_item',
		    [
                'label' => esc_html__('Hover' , 'bili-core'),
            ]
	    );

	    $this->add_control(
		    'button_color_hover_item',
		    [
			    'label' => esc_html__('Color', 'bili-core'),
				'type' => Controls_Manager::COLOR,
				'dynamic' => ['active' => true],
			    'selectors' => [
				    '{{WRAPPER}} .showcase__link:hover ~ .showcase__button .wgl-showcase_button.icon-read-more i,
				    {{WRAPPER}} .showcase__link:hover ~ .showcase__button .wgl-showcase_button.icon-read-more span' => 'color: {{VALUE}};',
					'{{WRAPPER}} .wgl-showcase_button.icon-read-more:hover i,
                    {{WRAPPER}} .wgl-showcase_button.icon-read-more:hover span' => 'color: {{VALUE}};',
			    ],
		    ]
	    );

	    $this->end_controls_tab();
        $this->end_controls_tabs();
        $this->end_controls_section();
    }

    protected function add_hidden_device_controls() {
		// The 'Hide On X' controls are displayed from largest to smallest, while the method returns smallest to largest.
		$active_devices = Plugin::$instance->breakpoints->get_active_devices_list( [ 'reverse' => true ] );
		$active_breakpoints = Plugin::$instance->breakpoints->get_active_breakpoints();

		foreach ( $active_devices as $breakpoint_key ) {
			$label = 'desktop' === $breakpoint_key ? esc_html__( 'Desktop', 'bili-core' ) : $active_breakpoints[ $breakpoint_key ]->get_label();

			$this->add_control(
				'wgl_hide_' . $breakpoint_key,
				[
					/* translators: %s: Device Name. */
					'label' => sprintf( esc_html__( 'Hide On %s', 'bili-core' ), $label ),
					'type' => Controls_Manager::SWITCHER,
					'default' => '',
					'label_on' => 'Hide',
					'label_off' => 'Show',
					'condition' => ['showcase_layout' => 'slide'],
					'return_value' => 'hidden-' . $breakpoint_key,
				]
			);
		}
	}

    protected function render()
    {
        extract($this->get_settings_for_display());

        if (isset($cursor_tooltip) && '' != $cursor_tooltip) {
            add_filter( 'wgl/bili_module_cursor', function () { return true; });
        }

        // Variables validation
        $img_size_string = $img_size_string ?? '';
        $img_size_array = $img_size_array ?? [];
        $img_aspect_ratio = $img_aspect_ratio ?? '';

        // Build structure
        $items_html_title = $items_html_image = $showcase__wrapper = '';

        $cursor = new WGL_Cursor;
        $cursor_data = $cursor->build($this, $this->get_settings_for_display());

        // The 'Hide On X' controls are displayed from largest to smallest, while the method returns smallest to largest.
		$active_devices = Plugin::$instance->breakpoints->get_active_devices_list( [ 'reverse' => true ] );

        $this->add_render_attribute(
		    'showcase_image_class',
		    [
			    'class' => [
				    'showcase__images',
			    ],
		    ]
	    );

        foreach ( $active_devices as $breakpoint_key ) {
            if(${'wgl_hide_' . $breakpoint_key}){
                $this->add_render_attribute(
                    'showcase_image_class',
                    [
                        'class' => [
                            'wgl-' . ${'wgl_hide_' . $breakpoint_key},
                        ],
                    ]
                );
            }
		}

        $kses_allowed_html = [
            'a' => [
                'href' => true, 'title' => true,
                'class' => true, 'style' => true,
                'rel' => true, 'target' => true
            ],
            'br' => ['class' => true, 'style' => true],
            'em' => ['class' => true, 'style' => true],
            'strong' => ['class' => true, 'style' => true],
            'span' => ['class' => true, 'style' => true],
            'small' => ['class' => true, 'style' => true],
            'p' => ['class' => true, 'style' => true],
            'i' => ['class' => true, 'style' => true],
        ];

        foreach ($items as $index => $item) {
            // Fields validation
            $thumbnail = $item['thumbnail'] ?? '';
            $attachment = get_post($thumbnail['id']);
            $image_data = wp_get_attachment_image_src($thumbnail['id'], 'full');
            $title_1st = $item['title'] ?? '';
            $subtitle = $item['subtitle'] ?? '';
            $category = $item['category'] ?? '';
            $link = $item['link'] ?? '';

            $has_link = !empty($link['url']);

            //* Image size
            $dim = null;
            $title = $image = '';

            if ($image_data) {
                $dim = WGL_Elementor_Helper::get_image_dimensions(
                    $img_size_array ?: $img_size_string,
                    $img_aspect_ratio,
                    $image_data
                );
            }

            if ($has_link) {
                $link = $this->get_repeater_setting_key('link', 'items', $index);
                $this->add_link_attributes($link, $item['link']);
            }

            if($dim){
                $image_url = aq_resize($image_data[0], $dim['width'], $dim['height'], true, true, true) ?: $image_data[0];
                //* Image Attachment
                $image_arr = [
                    'image' => $image_data[0],
                    'src' => $image_url,
                    'alt' => get_post_meta($thumbnail['id'], '_wp_attachment_image_alt', true),
                    'title' => $attachment->post_title,
                ];

                $this->add_render_attribute('image' . $index, [
                    'class' => 'image',
                    'src' => $image_arr['src'],
                    'alt' => $image_arr['alt']
                ]);
            }

            $image .=  '<div class="showcase__image"'.($dim && 'slide' === $showcase_layout ? ' style="background-image:url('.esc_url($image_arr['src']).')"' : '').'>';
                $image .=  $dim ? '<img '. $this->get_render_attribute_string('image' . $index). '>' : '';
            $image .= '</div>';

            $items_html_image .= $image;

            switch ($showcase_layout) {
                case 'fade_bg':
                case 'slide':
                    $items_html_title .= '<div class="showcase__title'.( isset($cursor_tooltip) && $cursor_tooltip ? ' wgl-cursor-text' : '' ).'"' . $cursor_data . '>'
                    . '<div class="showcase__title-inner">'
                    . ($has_link ? '<a ' . $this->get_render_attribute_string($link) . '>' : '')
                    . (!empty($subtitle) ? '<span class="subtitle">'. wp_kses($subtitle, $kses_allowed_html) .'</span>' : '')
                    . (!empty($category) ? '<span class="category">'. wp_kses($category, $kses_allowed_html) .'</span>' : '')
                    . (!empty($title_1st) ? '<' . esc_attr($title_tag) . ' class="title">'. wp_kses($title_1st, $kses_allowed_html) .'</' . esc_attr($title_tag) . '>' : '')
                    . ($has_link ? '</a>'  : '')
                    . $this->read_more_button($index, $has_link, $link, false)
                    . '</div>'
                    . '</div>';

                    break;

                case 'interactive':
                    $title = '<div class="showcase__title'.( isset($cursor_tooltip) && $cursor_tooltip ? ' wgl-cursor-text' : '' ).'"' . $cursor_data . '>'
                    . (!empty($subtitle) ? '<span class="subtitle">'. wp_kses($subtitle, $kses_allowed_html) .'</span>' : '')
                    . '<div class="showcase__title-inner">'
                    . (!empty($category) ? '<span class="category">'. wp_kses($category, $kses_allowed_html) .'</span>' : '')
                    . (!empty($title_1st) ? '<' . esc_attr($title_tag) . ' class="title">'. wp_kses($title_1st, $kses_allowed_html) .'</' . esc_attr($title_tag) . '>' : '')
                    . '</div>'
                    . '</div>';

                    $showcase__wrapper .= '<div class="showcase__content">';
                    if ($add_item_link && $has_link) {
                        $showcase__wrapper .= '<a class="showcase__link" ' . $this->get_render_attribute_string($link) . '></a>';
                    }
                    $showcase__wrapper .= $title;
                    $showcase__wrapper .= $image;
                    $showcase__wrapper .= $this->read_more_button($index, $has_link, $link, $add_item_link);
                    $showcase__wrapper .= '</div>';
                    break;
            }
        }

        if ('slide' === $showcase_layout) {
            echo '<div class="wgl-showcase slide-showcase">',
                '<div class="showcase__wrapper">',
                    '<div class="showcase__titles">',
                        $items_html_title,
                    '</div>',
                    '<div ',$this->get_render_attribute_string( 'showcase_image_class' ),'>',
                        $items_html_image,
                    '</div>',
                '</div>',
            '</div>';
        } elseif ('fade_bg' === $showcase_layout) {
            echo '<div class="wgl-showcase fade_bg-showcase">',
                '<div class="showcase__wrapper">',
                    '<div class="showcase__titles">',
                        $items_html_title,
                    '</div>',
                    '<div ',$this->get_render_attribute_string( 'showcase_image_class' ),'>',
                        $items_html_image,
                    '</div>',
                '</div>',
            '</div>';
        } else {
            echo '<div class="wgl-showcase interactive-showcase">',
                $showcase__wrapper,
            '</div>';
        }
    }

    public function read_more_button($index, $has_link, $link_item, $add_item_link_item)
    {
        extract($this->get_settings_for_display());

        $s_button = '';
        // Read more button
        if ($add_read_more) {
            $this->add_render_attribute('btn' . $index, 'class', ['wgl-showcase_button', 'icon-read-more']);

            $s_button = '<div class="showcase__button">';
            $s_button .= sprintf('<%s %s %s>',
                $add_item_link_item ? 'div' : 'a',
                !$add_item_link_item && $has_link ? $this->get_render_attribute_string($link_item) : '',
                $this->get_render_attribute_string('btn' . $index)
            );
            $s_button .= '<i class="icon flaticon flaticon-next"></i>';
            $s_button .= $add_item_link_item ? '</div>' : '</a>';
            $s_button .= '</div>';
        }

        return $s_button;
    }

    public function wpml_support_module() {
        add_filter( 'wpml_elementor_widgets_to_translate',  [$this, 'wpml_widgets_to_translate_filter']);
    }

    public function wpml_widgets_to_translate_filter( $widgets ){
        return \WGL_Extensions\Includes\WGL_WPML_Settings::get_translate(
            $this, $widgets
        );
    }
}
